/**
 * Copyright (c) 2019, Sinlmao (888@1st.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.ssl;

import javax.net.ssl.X509KeyManager;
import java.net.Socket;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.Vector;

/**
 * @author Sinlmao
 * program Sinlmao Commons Network Utils
 * description
 * create 2020-06-18 14:45
 */
public final class ProtocolSSLKeyManager implements X509KeyManager {

    protected String alias;
    protected KeyStore keyStore;
    protected char[] password;

    private final String algorithm;
    private final String issuer;

    public static ProtocolSSLKeyManager getInstance(KeyStore keyStore, char[] password) {
        return new ProtocolSSLKeyManager(keyStore, null, password);
    }

    public static ProtocolSSLKeyManager getInstance(KeyStore keyStore, String alias, char[] password) {
        return new ProtocolSSLKeyManager(keyStore, alias, password);
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    private ProtocolSSLKeyManager(KeyStore keyStore, String alias, char[] password) {
        this.keyStore = keyStore;
        this.alias = alias;
        this.password = password;
        try {
            Certificate c = keyStore.getCertificate(alias);
            algorithm = c.getPublicKey().getAlgorithm();
            issuer = ((X509Certificate) c).getIssuerDN().getName();
        } catch (Exception e) {
            throw new IllegalArgumentException(alias + " has a bad key");
        }
    }

    @Override
    public String[] getClientAliases(String algorithm, Principal[] principals) {
        String[] result;
        String alias = chooseClientAlias(new String[]{algorithm}, principals, null);
        if (alias == null)
            result = new String[0];
        else {
            result = new String[1];
            result[0] = alias;
        }
        return result;
    }

    @Override
    public String chooseClientAlias(String[] algorithms, Principal[] principals, Socket socket) {
        if (alias != null && !alias.isEmpty()) {
            return alias;
        }
        for (String algorithm : algorithms) {
            if (algorithm.equals(this.algorithm)) {
                if (principals == null) {
                    return alias;
                }
                for (Principal principal : principals) {
                    if (issuer.equals(principal.getName()))
                        return alias;
                }
            }
        }
        return null;
    }

    @Override
    public String[] getServerAliases(String algorithm, Principal[] principals) {
        return getClientAliases(algorithm, principals);
    }

    @Override
    public String chooseServerAlias(String algorithm, Principal[] principals, Socket socket) {
        return chooseClientAlias(new String[]{algorithm}, principals, socket);
    }

    @Override
    @SuppressWarnings({"rawtypes", "unchecked"})
    public X509Certificate[] getCertificateChain(String alias) {
        try {
            Certificate[] certificates = keyStore.getCertificateChain(alias);
            Vector vector = new Vector(certificates.length);
            Collections.addAll(vector, certificates);
            return (X509Certificate[]) vector.toArray(new X509Certificate[0]);
        } catch (KeyStoreException e) {
            return null;
        }
    }

    @Override
    public PrivateKey getPrivateKey(String alias) {
        try {
            return (PrivateKey) keyStore.getKey(alias, password);
        } catch (Exception e) {
            return null;
        }
    }
}