package cn.signit.sdk.type;

/**
 * 业务场景模板状态枚举类型.
 * </p>
 * 
 * @since 2.7.0
 */
public enum BstStatus {
    /**
     * 未知.
     * 
     * @since 2.7.0
     */
    UNKNOWN(-1, "未知"),
    /**
     * 草稿.
     * 
     * @since 2.7.0
     */
    DRAFT(0, "草稿"),
    /**
     * 启用.
     * 
     * @since 2.7.0
     */
    ENABLE(1, "启用"),
    /**
     * 停用.
     * 
     * @since 2.7.0
     */
    DISABLE(2, "停用"),
    /**
     * 草稿时删除到回收站.
     * 
     * @since 2.7.0
     */
    RECYCLED_FROM_DRAFT(BstStatus.DRAFT.getCode() + 100, "草稿时删除到回收站"),
    /**
     * 启用时删除到回收站.
     * 
     * @since 2.7.0
     */
    RECYCLED_FROM_ENABLE(BstStatus.ENABLE.getCode() + 100, "启用时删除到回收站"),
    /**
     * 停用时删除到回收站.
     * 
     * @since 2.7.0
     */
    RECYCLED_FROM_DISABLE(BstStatus.DISABLE.getCode() + 100, "停用时删除到回收站"),
    /**
     * 彻底删除不可恢复.
     * 
     * @since 2.7.0
     */
    DELETE_NOT_RECOVER(200, "彻底删除不可恢复");
    private final int code;
    private final String description;

    private BstStatus(int code, String description) {
        this.code = code;
        this.description = description;
    }

    /**
     * 
     * 根据int值来解析到对应的枚举实例.
     *
     * @param code
     *            code值
     * @return 对应的枚举实例.解析失败则返回<code>BstStatusType.UNKNOWN</code>
     * @since 2.7.0
     */
    public static BstStatus parse(int code) {
        BstStatus[] values = BstStatus.values();
        for (BstStatus bstType : values) {
            if (bstType.getCode() == code) {
                return bstType;
            }
        }
        return BstStatus.UNKNOWN;
    }

    /**
     * 
     * 根据String值来解析到对应的枚举实例.
     *
     * @param des
     *            枚举对应的描述
     * @return 对应的枚举实例.解析失败则返回<code>BstStatusType.UNKNOWN</code>
     * @since 2.7.0
     */
    public static BstStatus parse(String des) {
        BstStatus[] values = BstStatus.values();
        for (BstStatus bstType : values) {
            if (bstType.getDescription().equalsIgnoreCase(des)) {
                return bstType;
            }
        }
        return BstStatus.UNKNOWN;
    }

    /**
     * 获取code属性字段的值.
     * 
     * @since 2.7.0
     * @return 类型为int的code属性字段的值.
     */
    public int getCode() {
        return code;
    }

    /**
     * 获取description属性字段的值.
     * 
     * @since 2.7.0
     * @return 类型为String的description属性字段的值.
     */
    public String getDescription() {
        return description;
    }

    /**
     * 
     * 转换字符串中(比如：filters)包含的枚举英文为code.
     *
     * @param str
     *            需要转化的字符串
     * @return 转换完成的字符串
     * @since 2.7.0
     */
    public static String transferFilters(String str) {
        if (str == null) {
            return null;
        }
        String result = str;
        for (BstStatus val : values()) {
            if (result.contains("=" + val.toString())) {
                result = result.replace("=" + val.toString(), "=" + Integer.valueOf(val.getCode()));
            }
        }
        return result;
    }
}
