package cn.shazhengbo.kafka.storage.service;

import cn.shazhengbo.kafka.annotation.KafkaAckType;
import cn.shazhengbo.kafka.annotation.KafkaMessageState;
import cn.shazhengbo.kafka.annotation.KafkaMessageType;
import cn.shazhengbo.kafka.store.AbstractMessageStore;
import cn.shazhengbo.kafka.storage.entity.MessageConsumerGroup;
import cn.shazhengbo.kafka.storage.entity.MessageEventEntery;
import cn.shazhengbo.kafka.storage.mapper.IMessageConsumerGroupMapper;
import cn.shazhengbo.kafka.storage.mapper.IMessageEventEnteryMapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;

/**
 * @author CrazyShaQiuShi
 * @version 1.0
 * @description 消息存储服务类
 * @date 2020/8/22 15:47
 */
@Component
public class EventMessageStore extends AbstractMessageStore {
    @Autowired
    private IMessageEventEnteryMapper iMessageEventEnteryMapper;
    @Autowired
    private IMessageConsumerGroupMapper consumerGroupMapper;

    @Override
    public void store(String topic, int retries, KafkaMessageType type, KafkaAckType ackType, String message) {
        MessageEventEntery entery = new MessageEventEntery();
        entery.created(topic, retries, type, ackType, message);
        iMessageEventEnteryMapper.insert(entery);

    }

    @Override
    public void sendState(String topic, String uuid, KafkaMessageState state) {
        MessageEventEntery entery = iMessageEventEnteryMapper.selectOne(new QueryWrapper<MessageEventEntery>()
                .eq("topic", topic)
                .eq("uuid_", uuid).last(" limit 0,1"));
        if (entery != null) {
            entery.modified(state);
            iMessageEventEnteryMapper.updateById(entery);
        }
    }

    @Override
    public void retryMessage(long offSet, String uuid, String topic, String consumerGroup, long maxAckCount) {
        MessageEventEntery entery = iMessageEventEnteryMapper.selectOne(new QueryWrapper<MessageEventEntery>()
                .eq("uuid_", uuid)
                .eq("topic", topic)
                .last(" limit 0,1"));
        if (entery != null) {
            /**
             * 存在,消息中该组不包含未发送的记录
             */
            long unSendCount = countUnSend(consumerGroup, uuid, topic, entery.getParentId());
            System.out.println(unSendCount);
            if (entery != null && unSendCount <= 0) {
                long count = iMessageEventEnteryMapper.selectCount(new QueryWrapper<MessageEventEntery>()
                        .eq("parent_id", entery.getId())
                        .eq("consumer_group_", consumerGroup));
                /**
                 * 未超出最大重新确认次数,生成新的重试消息
                 */
                if (count <= maxAckCount) {
                    entery.retry(consumerGroup, count);
                    iMessageEventEnteryMapper.insert(entery);
                }
                /**
                 * 直接更改状态,标记出已处理此消息
                 */
                MessageConsumerGroup group = consumerGroupMapper.selectOne(new QueryWrapper<MessageConsumerGroup>()
                        .eq("uuid_", uuid)
                        .eq("consumer_group_", consumerGroup)
                        .last("limit 0,1"));
                if (group == null) {
                    group = new MessageConsumerGroup();
                    group.fail(offSet, uuid, consumerGroup, entery.getNextSendTime());
                    consumerGroupMapper.insert(group);
                } else {
                    group.setOffSet_(offSet);
                    group.setNextSendTime(entery.getNextSendTime());
                    consumerGroupMapper.updateById(group);
                }
            }
        }

    }

    /**
     * 获取当前消息,重试或者首次处理后未发的消息
     *
     * @param consumerGroup
     * @param uuid
     * @param topic
     * @param parentId
     * @return
     */
    private int countUnSend(String consumerGroup, String uuid, String topic, long parentId) {
        return iMessageEventEnteryMapper.countUnsend(consumerGroup, uuid, topic, parentId);
    }

    @Override
    public void ackDeal(String topic, long offSet, String uuid, String consumerGroup, KafkaAckType ackType) {
        /**
         * 获取当前uuid,的消息组记录值，若存在则更为处理成功
         */
        if (KafkaAckType.AUTO.equals(ackType)) {
            /**
             * 直接更改状态,标记出已处理此消息
             */
            MessageConsumerGroup group = consumerGroupMapper.selectOne(new QueryWrapper<MessageConsumerGroup>()
                    .eq("uuid_", uuid)
                    .eq("consumer_group_", consumerGroup)
                    .last("limit 0,1"));
            if (group == null) {
                group = new MessageConsumerGroup();
                group.ack(offSet, uuid, consumerGroup);
                consumerGroupMapper.insert(group);
            } else {
                group.setOffSet_(offSet);
                group.setNextSendTime(LocalDateTime.now());
                consumerGroupMapper.updateById(group);
            }
        } else if (KafkaAckType.RPC.equals(ackType)) {
            /**
             * 远程确认次消息
             */
        }
        /**
         * 清除,之前多次请求未处理的组信息
         */
        consumerGroupMapper.delete(new QueryWrapper<MessageConsumerGroup>()
                .eq("uuid_", uuid)
                .eq("consumer_group_", consumerGroup)
                .eq("state", 0));

    }
}
