package cn.sexycode.springo.form.manager.impl;

import cn.sexycode.springo.bo.model.BaseBoDef;
import cn.sexycode.springo.bo.permission.PermissionCalc;
import cn.sexycode.springo.core.base.core.json.JSONArray;
import cn.sexycode.springo.core.base.core.json.JSONObject;
import cn.sexycode.springo.core.base.core.util.BeanUtils;
import cn.sexycode.springo.core.base.core.util.string.StringUtil;
import cn.sexycode.springo.core.data.db.id.UniqueIdUtil;
import cn.sexycode.springo.core.data.db.manager.impl.BaseManagerImpl;
import cn.sexycode.springo.form.dao.FormDefDao;
import cn.sexycode.springo.form.dao.FormFieldDao;
import cn.sexycode.springo.form.dao.FormRightDao;
import cn.sexycode.springo.form.manager.FormManager;
import cn.sexycode.springo.form.manager.FormRightManager;
import cn.sexycode.springo.form.model.FormDef;
import cn.sexycode.springo.form.model.FormField;
import cn.sexycode.springo.form.model.FormRight;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * <pre>
 * 描述：BPM_FORM_RIGHT 处理实现类
 * </pre>
 */
@Service("bpmFormRightManager")
public class FormRightManagerImpl extends BaseManagerImpl<FormRight> implements FormRightManager {
    @Resource
    FormRightDao formRightDao;
    @Resource
    FormDefDao formDefDao;
    @Resource
    FormManager formManager;
    @Resource
    FormFieldDao formFieldDao;
    @Resource
    PermissionCalc permissionCalc;


    /**
     * 根据表单key获取默认的权限设置json。
     *
     * @param formKey    这个字段对应BPM_FROM_DEF的key字段。
     * @param isInstance 是否实例表单
     * @return
     */
    @Override
    public JSONObject getDefaultByFormDefKey(String formKey, boolean isInstance) {
        FormDef formDef = formDefDao.getByKey(formKey);
        String formDefId = formDef.getId();
        //b.type_,c.desc_,c.name_
        List<Map<String, String>> entList = formDefDao.getEntInfoByFormId(formDefId);
        //实体map列表。
        List<Map<String, String>> lowList = convertMapLower(entList);

        //只获取字段名称描述和实体名称。
        //A.name_,B.NAME_ ENT_NAME,A.desc_
        List<FormField> fieldList = formFieldDao.getExtByFormId(formDefId);
        //字段按照表进行分组。
        Map<String, List<FormField>> fieldMap = convertFormGroup(fieldList);

        JSONObject jsonObj = JSONObject.newJSONObject();

        //构建表的JSON
        JSONObject tableJson = JSONObject.newJSONObject();
        for (Map<String, String> entMap : lowList) {
            JSONObject json = buildTable(entMap, fieldMap, isInstance);
            if (json == null) continue;
            String entName = entMap.get("name_");
            tableJson.put(entName, json);
        }
        jsonObj.put("table", tableJson);

        //构建意见的JSON。
        String opinionJson = formDef.getOpinionConf();
        JSONObject opinionJsonObj = buildOpinion(opinionJson, isInstance);
        if (opinionJsonObj != null) {
            jsonObj.put("opinion", opinionJsonObj);
        }

        return jsonObj;
    }

    /**
     * {"bumenjingli": {
     * "description": "部门经理",
     * "read": [
     * {
     * "type": "everyone"
     * }
     * ],
     * "write": [
     * {
     * "type": "none"
     * }
     * ],
     * "required": [
     * {
     * "type": "none"
     * }
     * ]
     * },
     * "caiwu": {
     * "description": "财务意见",
     * "read": [
     * {
     * "type": "everyone"
     * }
     * ],
     * "write": [
     * {
     * "type": "none"
     * }
     * ],
     * "required": [
     * {
     * "type": "none"
     * }
     * ]
     * }
     * }
     *
     * @param json
     * @return
     */
    private JSONObject buildOpinion(String json, boolean isInstance) {
        if (StringUtil.isEmpty(json)) {
            return null;
        }

        JSONObject rtnJson = JSONObject.newJSONObject();

        JSONArray ary = JSONArray.parseArray(json);
        for (Object obj : ary) {
            JSONObject jsonObj = (JSONObject) obj;
            String name = jsonObj.getString("name");
            String desc = jsonObj.getString("description");

            JSONObject permissionJson = getPermissionJson(desc);

            if (isInstance) {
                permissionJson = getInstPermissionJson(desc);
            } else {
                permissionJson = getPermissionJson(desc);
            }

            rtnJson.put(name, permissionJson);

        }

        return rtnJson;
    }

    /**
     * 构建JSON格式如下:
     * {
     * "description": "个人简历",
     * "fields":{
     * "name": {
     * "description": "姓名",
     * "read": [
     * {
     * "type": "everyone"
     * }
     * ],
     * "write": [
     * {
     * "type": "none"
     * }
     * ],
     * "required": [
     * {
     * "type": "none"
     * }
     * ]
     * },
     * "age": {
     * "description": "年龄",
     * "read": [
     * {
     * "type": "everyone"
     * }
     * ],
     * "write": [
     * {
     * "type": "none"
     * }
     * ],
     * "required": [
     * {
     * "type": "none"
     * }
     * ]
     * }
     * }
     * ,
     * "main": true
     * }
     *
     * @param entMap
     * @param fieldMap
     * @param isInstance
     * @return
     */
    private JSONObject buildTable(Map<String, String> entMap, Map<String, List<FormField>> fieldMap, boolean isInstance) {

        String entName = entMap.get("name_");
        String entDesc = entMap.get("desc_");
        String type = entMap.get("type_");

        List<FormField> list = fieldMap.get(entName);
        if (BeanUtils.isEmpty(list)) {
            return null;
        }

        JSONObject jsonObj = JSONObject.newJSONObject();

        jsonObj.put("description", entDesc);
        //主表类型。
        if (BaseBoDef.BoEntRelation.MAIN.equalsIgnoreCase(type)) {
            jsonObj.put("main", true);
        }
		/*
		 * "main": false,
        "rights": {
            "hidden": false,
            "add": true,
            "del": true,
            "required": true
        }*/
        else {
            JSONObject rightJson = JSONObject.newJSONObject();
            rightJson.put("hidden", false);
            rightJson.put("add", true);
            rightJson.put("del", true);
            rightJson.put("required", false);

            jsonObj.put("main", false);
            jsonObj.put("rights", rightJson);
        }

        //构建字段。
        JSONObject fieldsJson = JSONObject.newJSONObject();

        for (FormField field : list) {
            JSONObject permissonJson = null;
            if (!isInstance) {
                permissonJson = getPermissionJson(field.getDescription());
            } else {
                permissonJson = getInstPermissionJson(field.getDescription());
            }

            fieldsJson.put(field.getName(), permissonJson);
        }

        jsonObj.put("fields", fieldsJson);

        return jsonObj;
    }

    /**
     * 获取默认的权限。
     *
     * @param desc
     * @return
     */
    private JSONObject getPermissionJson(String desc) {
        String json = "{\"description\": \"" + desc + "\",\"read\": [{\"type\": \"everyone\"}],"
                + "\"write\": [{\"type\": \"everyone\"}],\"required\": [{\"type\": \"none\"}]}";

        return JSONObject.parseObject(json);
    }

    /**
     * 获取只读权限。
     *
     * @param desc
     * @return
     */
    private JSONObject getInstPermissionJson(String desc) {
        String json = "{\"description\": \"" + desc + "\",\"read\": [{\"type\": \"everyone\"}]}";

        return JSONObject.parseObject(json);
    }


    /**
     * 将表单字段按照表进行分组。
     *
     * @param fieldList
     * @return
     */
    private Map<String, List<FormField>> convertFormGroup(List<FormField> fieldList) {
        Map<String, List<FormField>> map = new HashMap<String, List<FormField>>();

        for (FormField field : fieldList) {
            String entName = field.getEntName();
            if (map.containsKey(entName)) {
                List<FormField> list = map.get(entName);
                list.add(field);
            } else {
                List<FormField> list = new ArrayList<FormField>();
                list.add(field);
                map.put(entName, list);
            }
        }
        return map;
    }

    /**
     * map 转换 ，将map的键值转换成小写。
     *
     * @param entList
     * @return
     */
    private List<Map<String, String>> convertMapLower(List<Map<String, String>> entList) {
        List<Map<String, String>> rtnList = new ArrayList<Map<String, String>>();
        for (Map<String, String> row : entList) {
            Map<String, String> result = new HashMap<String, String>();
            for (Map.Entry<String, String> ent : row.entrySet()) {
                result.put(ent.getKey().toLowerCase(), ent.getValue());
            }
            rtnList.add(result);
        }
        return rtnList;
    }

    /**
     * 根据流程获取表单权限。
     *
     * @param flowKey
     * @param parentFlowKey
     * @return
     */
    private JSONObject getByFlowKey(String flowKey, String parentFlowKey) {

        FormRight right = formRightDao.getByFlowKey(flowKey, parentFlowKey, 1);
        if (right != null) {
            JSONObject rtnJson = JSONObject.parseObject(right.getPermission());
            return rtnJson;
        }
        return null;
    }

    /**
     * 根据节点获取表单权限。
     *
     * @param flowKey
     * @param nodeId
     * @param parentFlowKey
     * @return
     */
    private JSONObject getByFlowNodeId(String flowKey, String nodeId, String parentFlowKey) {
        FormRight right = formRightDao.getByFlowNodeId(flowKey, nodeId, parentFlowKey);
        if (right != null) {
            JSONObject rtnJson = JSONObject.parseObject(right.getPermission());
            return rtnJson;
        }
        return null;
    }

    @Override
    public JSONObject getByFormKey(String formKey, boolean isReadOnly) {
        FormRight right = formRightDao.getByFormKey(formKey, isReadOnly);
        JSONObject rtnJson = null;
        if (right != null) {
            rtnJson = JSONObject.parseObject(right.getPermission());
            return rtnJson;
        } else {
            String formMetaKey = formDefDao.getMetaKeyByFormKey(formKey);
            if (StringUtil.isEmpty(formMetaKey)) {
                return null;//没有找到formKey 则返回空吧。
            }

            rtnJson = getDefaultByFormDefKey(formMetaKey, isReadOnly);
            return rtnJson;
        }
    }


    /**
     * 根据流程定义获取实例权限配置。
     *
     * @param flowKey
     * @return
     */
    private JSONObject getByInst(String flowKey) {
        FormRight right = formRightDao.getByFlowKey(flowKey, "", 2);
        if (right != null) {
            JSONObject rtnJson = JSONObject.parseObject(right.getPermission());
            return rtnJson;
        }
        return null;
    }


    @Override
    public void removeInst(String flowKey) {
        formRightDao.removeByFlowKey(flowKey, "", 2);
    }

    /*
     * 如果流程表单，节点已经授权了表单， 但已经更换了表单， 则删除原来的
     *
     */
    @Override
    public void remove(String formKey, String flowKey, String nodeId,
                       String parentFlowKey) {
        FormRight right = null;
        if (StringUtil.isNotEmpty(flowKey)) {
            if (StringUtil.isNotEmpty(nodeId)) {
                right = formRightDao.getByFlowNodeId(flowKey, nodeId, parentFlowKey);
            } else {
                right = formRightDao.getByFlowKey(flowKey, parentFlowKey, 1);
            }
        }
        if (right != null && !formKey.equals(right.getFormKey())) {
            remove(flowKey, nodeId, parentFlowKey);
        }
    }

    @Override
    public void remove(String flowKey, String nodeId, String parentFlowKey) {
        formRightDao.removeByFlowNode(flowKey, nodeId, parentFlowKey);
    }


    @Override
    public void remove(String flowKey, String parentFlowKey) {
        formRightDao.removeByFlowKey(flowKey, parentFlowKey, 1);
    }


    @Override
    public void save(String formKey, String flowKey, String parentFlowKey, String nodeId, String permission, int type) {

        //清除之前的流程设置。
        if (StringUtil.isNotEmpty(flowKey)) {//表单已绑定流程
            if (type == 1) {
                if (StringUtil.isEmpty(nodeId)) {
                    remove(flowKey, parentFlowKey);
                } else {
                    remove(flowKey, nodeId, parentFlowKey);
                }
            } else {
                removeInst(flowKey);
            }
        } else {
            removeBusFormPermision(formKey, type);
        }

        //添加表单权限
        String id = UniqueIdUtil.getSuid();
        FormRight right = new FormRight();
        right.setId(id);

        right.setFormKey(formKey);
        right.setFlowKey(flowKey);
        right.setNodeId(nodeId);
        right.setParentFlowKey(parentFlowKey);
        right.setPermission(permission);
        right.setPermissionType(type);
        formRightDao.insert(right);
    }


    private void removeBusFormPermision(String formKey, int type) {
        formRightDao.removeBusFormPermision(formKey, type);
    }


    /**
     * formKey bpm_form的key字段。
     * 获取权限配置
     */
    @Override
    public JSONObject getPermissionSetting(String formKey, String flowKey, String parentFlowKey, String nodeId,
                                           int type, boolean isEditPermission) {
        boolean isReadOnly = type != 1;
        JSONObject json = null;

        //流程权限
        if (StringUtil.isNotEmpty(flowKey)) {
            if (type == 1) {
                if (StringUtil.isEmpty(nodeId)) {
                    json = getByFlowKey(flowKey, parentFlowKey);
                } else {
                    json = getByFlowNodeId(flowKey, nodeId, parentFlowKey);
                    //获取下全局的权限配置。
                    if (json == null && !isEditPermission) {
                        json = getByFlowKey(flowKey, parentFlowKey);
                    }
                }
            }
            //实例
            else {
                json = getByInst(flowKey);
            }

            //如果编辑情况，流程设置没取到就初始化新的
            if (isEditPermission && json == null) {
                String formMetaKey = formDefDao.getMetaKeyByFormKey(formKey);
                json = getDefaultByFormDefKey(formMetaKey, isReadOnly);
                return json;
            }
        }

        //如果未配置节点全局权限，获取表单配置权限,或者表单基础权限
        if (json == null) {
            json = getByFormKey(formKey, isReadOnly);
        }

        return json;
    }


    /**
     * 计算权限。
     * formKey : bpm_form的key。
     * <pre>
     * {
     *     "fields": {
     *         "table1": {
     *             "name": "w",
     *             "age": "b"
     *         },
     *         "table2": {
     *             "name": "w",
     *             "age": "r"
     *         },
     *         "table3": {
     *             "name": "w",
     *             "age": "w"
     *         }
     *     },
     *     "table":{
     *      "table1":{"hidden":true}
     *      "table2":{"hidden":false,"add":"true","del":"true","required":"true"}
     *     },
     *     "opinion":{"jzyj":"w","zxyj":"r","zxyj":"b"}
     * </pre>
     */
    @Override
    public JSONObject getPermission(String formKey, String flowKey, String parentFlowKey, String nodeId, int type) {

        JSONObject json = getPermissionSetting(formKey, flowKey, parentFlowKey, nodeId, type, false);

        return calcFormPermission(json);

    }

    /**
     * 通过获取的permissionJson 获取表单权限
     *
     * @param permissionConf
     * @return
     */
    @Override
    public JSONObject calcFormPermission(JSONObject permissionConf) {
        Map<String, Set<String>> profilesMap = permissionCalc.getCurrentProfiles();

        JSONObject rtnJson = JSONObject.newJSONObject();
        //获取表单权限设定。
        //获取表
        JSONObject tableJsons = permissionConf.getJSONObject("table");

        //1.构建字段权限JSON，构建子表权限。
        JSONObject rtnTableFieldJson = JSONObject.newJSONObject();
        JSONObject rtnTableJson = JSONObject.newJSONObject();

        for (Iterator<String> tableIt = tableJsons.keySet().iterator(); tableIt.hasNext(); ) {
            String tableName = tableIt.next();
            JSONObject tableJson = tableJsons.getJSONObject(tableName);
            JSONObject tableFieldJson = buildTablePermission(tableJson, profilesMap);
            rtnTableFieldJson.put(tableName, tableFieldJson);
            boolean isMain = tableJson.getBoolean("main");
            if (!isMain) {
                JSONObject tableRights = tableJson.getJSONObject("rights");
                rtnTableJson.put(tableName, tableRights);
            }
        }
        //字段权限。
        rtnJson.put("fields", rtnTableFieldJson);
        //2.构建子表权限JSON。
        if (rtnTableJson.size() > 0) {
            rtnJson.put("table", rtnTableJson);
        }
        if (!permissionConf.containsKey("opinion") || StringUtil.isEmpty(permissionConf.getString("opinion"))) {
            return rtnJson;
        }

        //3.构建意见权限JSON。
        JSONObject rtnOpinionJson = JSONObject.newJSONObject();
        JSONObject opinionJson = permissionConf.getJSONObject("opinion");
        for (Iterator<String> opinionIt = opinionJson.keySet().iterator(); opinionIt.hasNext(); ) {
            String opinionName = opinionIt.next();
            JSONObject perJson = opinionJson.getJSONObject(opinionName);
            //进行权限计算
            String permission = calcPermission(perJson, profilesMap);
            rtnOpinionJson.put(opinionName, permission);
        }
        rtnJson.put("opinion", rtnOpinionJson);

        return rtnJson;
    }

    /**
     * 字段1:权限，权限的值(n:没有权限,r:只读,w:编辑,b:必填)
     * 如果有必填就不再继续判断，没有就判断判断编辑权限，再判断只读权限，没有就没有权限。
     * 所有人权限和无权限只能有一项。
     *
     * @param tableJson
     * @return
     */
    private JSONObject buildTablePermission(JSONObject tableJson, Map<String, Set<String>> profilesMap) {
        JSONObject rtnJson = JSONObject.newJSONObject();
        JSONObject fieldJsons = tableJson.getJSONObject("fields");
        //对每一个字段计算权限。
        for (Iterator<String> fieldIt = fieldJsons.keySet().iterator(); fieldIt.hasNext(); ) {
            String fieldName = fieldIt.next();
            JSONObject perJson = fieldJsons.getJSONObject(fieldName);
            String permission = calcPermission(perJson, profilesMap);
            rtnJson.put(fieldName, permission);
        }
        return rtnJson;
    }

    /**
     * 权限计算。
     * <pre>
     * 1.先判断是否有必填权限。
     * 2.再判断是否有编辑权限。
     * 3.再判断是否有读的权限。
     * 4.无权限。
     *
     * 权限的值：
     * n: 没有权限
     * b: 必填
     * w: 编辑
     * r: 只读
     * </pre>
     *
     * @param perJson
     * @return
     */
    private String calcPermission(JSONObject perJson, Map<String, Set<String>> profilesMap) {
        //判断必填权限
        boolean hasRequired = hasRight(perJson, "required", profilesMap);

        String permission = "n";

        if (hasRequired) {
            permission = "b";
        } else {
            //判断编辑
            boolean hasWrite = hasRight(perJson, "write", profilesMap);
            if (hasWrite) {
                permission = "w";
            } else {
                //判断只读
                boolean hasRead = hasRight(perJson, "read", profilesMap);
                if (hasRead) {
                    permission = "r";
                }
            }
        }
        return permission;
    }

    /**
     * 判断是否有权限。
     * <p>
     * 参数：
     * jsonObj: 格式如下
     * {
     * "description": "姓名",
     * "read": [
     * {
     * "type": "everyone"
     * }
     * ],
     * "write": [
     * {
     * "type": "none"
     * }
     * ],
     * "required": [
     * {
     * "type": "none"
     * }
     * ]
     * }
     *
     * @param jsonObj
     * @param type    可能的值required,write,read
     * @return
     */
    private boolean hasRight(JSONObject jsonObj, String type, Map<String, Set<String>> profilesMap) {
        if (!jsonObj.containsKey(type)) {
            return false;
        }

        JSONArray jsonArray = (JSONArray) jsonObj.get(type);
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject json = jsonArray.getJSONObject(i);

            boolean hasRight = permissionCalc.hasRight(json.toJSONString(), profilesMap);

            if (hasRight) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void removeByFormKey(String formKey) {
        formRightDao.removeByFormKey(formKey);
    }


}
