package cn.sexycode.springo.form.manager.impl;

import cn.sexycode.springo.core.base.core.exceptions.ModelExistException;
import cn.sexycode.springo.core.base.core.json.JSONObject;
import cn.sexycode.springo.core.base.core.util.BeanUtils;
import cn.sexycode.springo.core.data.db.id.UniqueIdUtil;
import cn.sexycode.springo.core.data.db.manager.impl.BaseManagerImpl;
import cn.sexycode.springo.form.dao.FormDefDao;
import cn.sexycode.springo.form.dao.FormFieldDao;
import cn.sexycode.springo.form.manager.FormDefManager;
import cn.sexycode.springo.form.manager.FormManager;
import cn.sexycode.springo.form.model.FormDef;
import cn.sexycode.springo.form.model.FormField;
import cn.sexycode.springo.form.model.IForm;
import cn.sexycode.springo.form.rest.vo.FormDefVO;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * 描述：form_def 处理实现类
 */
@Service("bpmFormDefManager")
public class FormDefManagerImpl extends BaseManagerImpl<FormDef> implements FormDefManager {
    @Resource
    FormDefDao formDefDao;
    @Resource
    FormFieldDao formFieldDao;
    @Resource
    FormManager formManager;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(FormDefVO formDefVO) {
        if (BeanUtils.isNotEmpty(formDefVO.getId())) {
            update(formDefVO);
        } else {
            String formKey = formDefVO.getKey();
            if (getByKey(formKey) != null) {
                throw new ModelExistException("表单已经存在！key:" + formKey);
            }

            FormDef formDef = getFormDef(formDefVO);
            super.create(formDef);
            createFields(formDef);
            formDefVO.setId(formDef.getId());
            updateFormBo(formDefVO);
        }

    }

    private FormDef getFormDef(FormDefVO formDefVO) {
        FormDef formDef = BeanUtils.copyProperties(FormDef.class, formDefVO);
        List<FormField> formFields = Optional.of(formDef.getFormFields()).orElse(new ArrayList<>());
        formDefVO.getTables().forEach(vo -> vo.getFields().forEach(formFieldVO -> {
            formFields.add(BeanUtils.copyProperties(FormField.class, formFieldVO));
        }));
        formDef.setFormFields(formFields);
        JSONObject expand = JSONObject.newJSONObject();
        expand.put("tables", formDefVO.getTables());
        expand.put("boDefList", formDefVO.getBoDefList());
        formDef.setExpand(expand.toJSONString());
        return formDef;
    }


    private void createFields(FormDef formDef) {
        formDef.getFormFields().forEach(formField -> {
            formField.setFormId(formDef.getId());
            formFieldDao.insert(formField);
        });

    }

    /**
     * 更新表单和BO的关联记录
     *
     * @param formDef
     */
    private void updateFormBo(FormDefVO formDef) {
        String formId = formDef.getId();
        formDef.getBoDefList().forEach(boDefVo -> {
            formDefDao.createFormBo(UniqueIdUtil.getSuid(), boDefVo.getId(), formId);
        });
    }


    /**
     * 获取实体,包含 字段信息，分组信息，主表表单
     */
    @Override
    public FormDef get(String entityId) {
        return super.get(entityId);
    }


    @Override
    public FormDef getByKey(String formKey) {
        return formDefDao.getByKey(formKey);
    }

    /**
     * 删除记录包含子表记录
     */
    @Override
    public void remove(String entityId) {
        List<IForm> forms = formManager.getByDefId(entityId);
        for (IForm form : forms) {
            formManager.remove(form.getId());
        }

        formDefDao.deleteFormBo(entityId);
        formFieldDao.delByMainId(entityId);

        super.remove(entityId);
    }

    /**
     * 更新实体同时更新子表记录
     */
    @Override
    public void update(FormDef bpmFormDef) {
        throw new IllegalStateException("此方法不支持");
    }


    @Override
    public List<FormDef> getByBODefId(String BODefId) {
        return formDefDao.getByBODefId(BODefId);
    }


    @Override
    public void updateOpinionConf(String id, String opinionJson) {
        formDefDao.updateOpinionConf(id, opinionJson);

    }


    @Override
    public String getMetaKeyByFormKey(String formKey) {
        return formDefDao.getMetaKeyByFormKey(formKey);
    }


    @Override
    public List<String> getBoCodeByFormId(String formDefId) {
        return formDefDao.getBOCodeByFormId(formDefId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(FormDefVO formDefVO) {
        FormDef formDef = getFormDef(formDefVO);

        super.update(formDef);
        String formDefId = formDef.getId();

        formFieldDao.delByMainId(formDefId);
        createFields(formDef);
        formDefDao.deleteFormBo(formDefId);
        updateFormBo(formDefVO);
    }

}
