package cn.sexycode.springo.data.boot.autoconfigure;

import cn.sexycode.springo.core.base.api.IEnum;
import cn.sexycode.springo.core.base.core.util.string.StringUtil;
import cn.sexycode.springo.core.data.db.DataConfiguration;
import cn.sexycode.springo.core.data.db.mybatis.PageInterceptor;
import cn.sexycode.springo.core.data.db.mybatis.dialect.MySQLDialect;
import cn.sexycode.springo.core.data.db.mybatis.handler.BooleanTypeHandler;
import cn.sexycode.springo.core.data.db.mybatis.handler.EnumTypeHandler;
import cn.sexycode.springo.core.data.db.mybatis.handler.EnumTypeRegister;
import cn.sexycode.springo.core.data.db.mybatis.support.SqlInjector;
import com.baomidou.mybatisplus.autoconfigure.ConfigurationCustomizer;
import com.baomidou.mybatisplus.autoconfigure.MybatisPlusAutoConfiguration;
import com.baomidou.mybatisplus.autoconfigure.MybatisPlusProperties;
import com.baomidou.mybatisplus.extension.plugins.OptimisticLockerInterceptor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.ibatis.type.TypeHandlerRegistry;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

/**
 * @author qinzaizhen
 */
@EnableAutoConfiguration
@Configuration
@Import(DataConfiguration.class)
@ConditionalOnClass({SqlSessionFactory.class, SqlSessionFactoryBean.class})
@EnableConfigurationProperties(DataProperties.class)
@AutoConfigureBefore(MybatisPlusAutoConfiguration.class)
@AutoConfigureAfter(DataSourceAutoConfiguration.class)
@Aspect
public class DataAutoConfiguration {
    private static final Logger LOGGER = LoggerFactory.getLogger(DataAutoConfiguration.class);
    private final DataProperties dataProperties;

    public DataAutoConfiguration(ApplicationContext applicationContext, DataProperties dataProperties) {
        this.dataProperties = dataProperties;
    }


    @Bean
    @ConditionalOnMissingBean
    public SqlInjector sqlInjector() {
        return new SqlInjector();
    }

    @Pointcut("execution(* com.baomidou.mybatisplus.autoconfigure.MybatisPlusAutoConfiguration.sqlSessionFactory(..))")
    public void mybatisPlusConfigPointcut() {
    }

    /**
     * 由于mybatis-plus 在初始化sessionFactoryBean时无法修改,通过aop将框架的mapper文件注入到它的配置属性中,注入框架mapper文件
     *
     * @param joinPoint
     */
    @Before("mybatisPlusConfigPointcut()")
    public void injectMapperLocation(JoinPoint joinPoint) {
        Object target = joinPoint.getTarget();
        if (target instanceof MybatisPlusAutoConfiguration) {
            if (dataProperties.isInjectNestMapper()) {
                Field field = ReflectionUtils.findField(MybatisPlusAutoConfiguration.class, "properties");
                if (field != null) {
                    ReflectionUtils.makeAccessible(field);
                    Object properties = ReflectionUtils.getField(field, target);
                    if (properties != null) {
                        MybatisPlusProperties mybatisPlusProperties = (MybatisPlusProperties) properties;
                        String[] mapperLocations = mybatisPlusProperties.getMapperLocations();
                        List<String> mapperLocationList = new ArrayList<>();
                        mapperLocationList.add("classpath*:framework/mapper/*.mapper.xml");
                        if (mapperLocations != null) {
                            mapperLocationList.addAll(Arrays.asList(mapperLocations));
                        }
                        mybatisPlusProperties.setMapperLocations(mapperLocationList.toArray(new String[0]));
                    } else {
                        LOGGER.warn("未读取到MybatisPlusAutoConfiguration 属性 properties的值");
                    }
                } else {
                    LOGGER.warn("未读取到MybatisPlusAutoConfiguration 的属性 properties");
                }
            } else {
                LOGGER.info("忽略注入框架mapper");
            }
        }
    }

    @Bean
    public ConfigurationCustomizer customizer(List<EnumTypeRegister> enumTypeRegisters) {
        return configuration -> {
            TypeHandlerRegistry typeHandlerRegistry = configuration.getTypeHandlerRegistry();
            try {
                typeHandlerRegistry.register(Boolean.class.getName(), BooleanTypeHandler.class.getCanonicalName());
                typeHandlerRegistry.register(IEnum.class.getName(), EnumTypeHandler.class.getCanonicalName());

                if (CollectionUtils.isNotEmpty(enumTypeRegisters)) {
                    enumTypeRegisters.forEach(r -> typeHandlerRegistry.register(r.getJavaType(), r.getResultTypeHandler()));
                }
            } catch (ClassNotFoundException e) {
                LOGGER.warn("注册类型转换器失败: ", e);
            }

        };
    }

    /**
     * 分页插件
     */
    @Bean
    public PageInterceptor paginationInterceptor() {
        PageInterceptor pageInterceptor = new PageInterceptor();
        Properties properties = new Properties();
        properties.put("Dialect.mysql", MySQLDialect.class.getName());
        if (StringUtil.isNotZeroEmpty(this.dataProperties.getDbType())) {
            properties.put("dbType", dataProperties.getDbType());
        } else {
            properties.put("dbType", "mysql");
        }
        pageInterceptor.setProperties(properties);
        return pageInterceptor;
    }

    @Bean
    public OptimisticLockerInterceptor optimisticLockerInterceptor() {
        return new OptimisticLockerInterceptor();
    }
}
