package cn.schoolwow.quickserver.util;

import cn.schoolwow.quickserver.domain.Client;
import cn.schoolwow.quickserver.exception.HttpStatusException;
import cn.schoolwow.quickserver.response.HttpStatus;
import com.alibaba.fastjson.JSON;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.net.SocketException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Map;
import java.util.StringTokenizer;

public class QuickServerUtil {
    private static final ThreadLocal<ByteArrayOutputStream> threadLocal = new ThreadLocal<>();

    /**
     * 处理请求过程中产生的异常
     * */
    public static void handleExceptionHandler(Client client, Throwable e) {
        if (e instanceof InvocationTargetException) {
            e = ((InvocationTargetException) e).getTargetException();
        }
        client.httpResponse.httpStatus(HttpStatus.INTERNAL_SERVER_ERROR);
        if( e instanceof HttpStatusException){
            HttpStatusException httpStatusException = (HttpStatusException) e;
            client.httpResponse.httpStatus(HttpStatus.getStatus(httpStatusException.getStatus()));
        }
        if(null == client.serverConfigMeta.exceptionHandler){
            if(!(e instanceof HttpStatusException)&&!(e instanceof SocketException)){
                e.printStackTrace();
            }
            client.httpResponse.status(HttpStatus.INTERNAL_SERVER_ERROR);
            client.httpResponseMeta.result = e.getMessage();
            if(null==client.httpResponseMeta.result){
                client.httpResponseMeta.result = "空指针异常!";
            }
        }else{
            Object result = client.serverConfigMeta.exceptionHandler.beforeBodyWrite(e, client.httpRequest, client.httpResponse, client.httpSession);
            if (null != result) {
                client.httpResponseMeta.result = client.httpResponseMeta.result = JSON.toJSONStringWithDateFormat(result,"yyyy-MM-dd HH:mm:ss");
            }
        }
        client.httpResponseMeta.contentType = "application/json";
        byte[] bytes = client.httpResponseMeta.result.getBytes(StandardCharsets.UTF_8);
        client.httpResponseMeta.contentLength = bytes.length;
        client.httpResponseMeta.bodyInputStream = new ByteArrayInputStream(bytes);
    }

    /**
     * 处理表单参数
     * @param body 内容
     * @param dataMap 参数存放表
     * @param charset 编码格式
     * */
    public static void handleFormData(String body, Map<String,String> dataMap, String charset) throws UnsupportedEncodingException {
        StringTokenizer st = new StringTokenizer(body, "&");
        while (st.hasMoreTokens()) {
            String token = st.nextToken();
            String key = token.substring(0,token.indexOf("="));
            String value = token.endsWith("=")?"":token.substring(token.indexOf("=")+1);
            dataMap.put(key, URLDecoder.decode(value, charset));
        }
    }

    /**
     * 读取一行
     *
     * @param inputStream 输入流
     * @param charset     编码格式
     * @return 读入内容, 为null表示流读取完毕
     */
    public static String readLine(InputStream inputStream, String charset) throws IOException {
        ByteArrayOutputStream baos = threadLocal.get();
        if (null == baos) {
            baos = new ByteArrayOutputStream();
            threadLocal.set(baos);
        }
        baos.reset();
        int b;
        while ((b = inputStream.read()) != -1) {
            if (b == 0x0D) {
                b = inputStream.read();
                if (b == 0x0A) {
                    break;
                } else {
                    baos.write(0x0D);
                    baos.write(b);
                }
            } else {
                baos.write(b);
            }
        }
        if (b == -1) {
            inputStream.close();
            return null;
        }
        baos.flush();
        return new String(baos.toByteArray(), charset);
    }

    /**
     * 字节数组转十六进制
     */
    public static String byteArrayToHex(byte[] bytes) {
        if(null==bytes||bytes.length==0){
            return "null";
        }
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(bytes[i] & 0xFF);
            if (hex.length() < 2) {
                sb.append(0);
            }
            sb.append(hex);
        }
        return sb.toString().toUpperCase();
    }

    /**
     * 十六进制字符串转字节数组
     */
    public static byte[] hexToByteArray(String hex) {
        if(null==hex||hex.isEmpty()){
            return null;
        }
        byte[] result = new byte[hex.length() / 2];
        for (int i = 0, j = 0; j < result.length; i += 2,j++) {
            result[j] = (byte) Integer.parseInt(hex.substring(i, i + 2), 16);
        }
        return result;
    }

    /**生成Web-Socket-Key随机值*/
    private static SecureRandom secWebSocketKeyRandom = new SecureRandom();

    /**
     * 生成随机字节数组
     * @param bytes 字节数组
     * */
    public static void randomBytes(byte[] bytes){
        secWebSocketKeyRandom.nextBytes(bytes);
    }

    /**
     * 计算SecWebSocketAccept的值
     * @param secWebSocketKey 请求头部Sec-WebSocket-Key的值
     * @return 响应头部Sec-WebSocket-Accept的值
     * */
    public static String calculateSecWebSocketAccept(String secWebSocketKey) throws NoSuchAlgorithmException {
        //计算Sec-WebSocket-Accept
        byte[] plainText = (secWebSocketKey+"258EAFA5-E914-47DA-95CA-C5AB0DC85B11").getBytes();
        byte[] shaEncoded = MessageDigest.getInstance("SHA").digest(plainText);
        String secWebSocketAccept = Base64.getEncoder().encodeToString(shaEncoded);
        return secWebSocketAccept;
    }
}
