package cn.schoolwow.quickserver.server;

import cn.schoolwow.quickserver.controller.ExceptionHandler;
import cn.schoolwow.quickserver.controller.RequestMethod;
import cn.schoolwow.quickserver.controller.ResponseBodyAdvice;
import cn.schoolwow.quickserver.controller.annotation.CrossOrigin;
import cn.schoolwow.quickserver.controller.annotation.RequestMapping;
import cn.schoolwow.quickserver.controller.annotation.RestController;
import cn.schoolwow.quickserver.domain.Client;
import cn.schoolwow.quickserver.domain.ControllerMeta;
import cn.schoolwow.quickserver.domain.HttpSessionMeta;
import cn.schoolwow.quickserver.domain.ServerConfigMeta;
import cn.schoolwow.quickserver.websocket.ServerEndpoint;
import cn.schoolwow.quickserver.websocket.WebSocketServerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.*;
import java.util.concurrent.TimeUnit;

public class HttpServer {
    Logger logger = LoggerFactory.getLogger(HttpServer.class);

    private ServerConfigMeta serverConfigMeta;

    public HttpServer(ServerConfigMeta serverConfigMeta) {
        this.serverConfigMeta = serverConfigMeta;
    }

    /**
     * 启动http服务
     * */
    public void start() throws IOException {
        registerController();
        registerWebsocket();
        startKeepAliveScheduleTask();
        startHttpSessionScheduleTask();
        ServerSocket serverSocket = new ServerSocket();
        InetSocketAddress inetSocketAddress = new InetSocketAddress(serverConfigMeta.port);
        serverSocket.bind(inetSocketAddress);
        logger.info("服务器启动:{}", inetSocketAddress);
        while (true) {
            final Socket socket = serverSocket.accept();
            logger.debug("接收客户端Socket链接:{}",socket);
            serverConfigMeta.threadPoolExecutor.execute(new HttpClientThread(socket,serverConfigMeta));
        }
    }

    /**
     * 注册Controller信息
     */
    private void registerController() {
        serverConfigMeta.applicationContext.refresh();
        String[] beanNames = serverConfigMeta.applicationContext.getBeanNamesForAnnotation(RestController.class);
        for (String beanName : beanNames) {
            Object instance = serverConfigMeta.applicationContext.getBean(beanName);
            String bathUrl = "";
            RequestMapping classRequestMapping = instance.getClass().getDeclaredAnnotation(RequestMapping.class);
            if (classRequestMapping != null) {
                bathUrl = classRequestMapping.value();
            }
            Method[] methods = instance.getClass().getMethods();
            for (Method method : methods) {
                RequestMapping methodRequestMapping = method.getAnnotation(RequestMapping.class);
                if (methodRequestMapping == null) {
                    continue;
                }
                ControllerMeta controllerMeta = new ControllerMeta();
                controllerMeta.instance = instance;
                controllerMeta.mappingUrl = bathUrl + methodRequestMapping.value();
                controllerMeta.requestMethodList.addAll(Arrays.asList(methodRequestMapping.method()));
                //跨域方法添加OPTION请求支持
                if (null != method.getAnnotation(CrossOrigin.class) || null != method.getDeclaringClass().getAnnotation(CrossOrigin.class)) {
                    controllerMeta.requestMethodList.add(RequestMethod.OPTIONS);
                }
                controllerMeta.antPatternUrl = controllerMeta.mappingUrl.replaceAll("\\{\\w+\\}", "\\*");
                controllerMeta.method = method;
                serverConfigMeta.controllerMetaList.add(controllerMeta);
                logger.info("映射路径 [{},method={}] onto {}", controllerMeta.mappingUrl, controllerMeta.requestMethodList, method);
            }
        }
        try {
            serverConfigMeta.responseBodyAdvice = serverConfigMeta.applicationContext.getBean(ResponseBodyAdvice.class);
        } catch (NoSuchBeanDefinitionException e) {
        }
        try {
            serverConfigMeta.exceptionHandler = serverConfigMeta.applicationContext.getBean(ExceptionHandler.class);
        } catch (NoSuchBeanDefinitionException e) {
        }
    }

    /**注册WebSocket*/
    private void registerWebsocket(){
        String[] beanNames = serverConfigMeta.applicationContext.getBeanNamesForType(WebSocketServerListener.class);
        for(String beanName:beanNames){
            WebSocketServerListener webSocketServerListener = (WebSocketServerListener) serverConfigMeta.applicationContext.getBean(beanName);
            ServerEndpoint serverEndpoint = webSocketServerListener.getClass().getAnnotation(ServerEndpoint.class);
            if(null==serverEndpoint){
                throw new IllegalArgumentException("WebSocket类未指定路径!请使用serverEndpoint注册指定路径!类名:"+webSocketServerListener.getClass().getName());
            }
            logger.info("映射WebSocket [{}] onto {}", serverEndpoint.value(), webSocketServerListener.getClass().getName());
            serverConfigMeta.uriWebSocketServerListenerMap.put(serverEndpoint.value(), webSocketServerListener);
        }
    }

    /**
     * 开启会话定时任务检查
     * */
    private void startHttpSessionScheduleTask(){
        serverConfigMeta.scheduledExecutorService.scheduleWithFixedDelay(() -> {
            Set<Map.Entry<String, HttpSessionMeta>> entrySet = serverConfigMeta.sessionMap.entrySet();
            entrySet.removeIf(entry -> (System.currentTimeMillis() - entry.getValue().lastAccessedTime) / 1000 >= serverConfigMeta.sessionTimeout);
        }, 60, 600, TimeUnit.SECONDS);
    }

    /**
     * 开启keepAlive定时任务检查
     * */
    private void startKeepAliveScheduleTask(){
        if(!serverConfigMeta.keepAlive){
            return;
        }
        serverConfigMeta.scheduledExecutorService.scheduleWithFixedDelay(() -> {
            logger.trace("开始进行keepAlive定时检查任务");
            synchronized (serverConfigMeta.keepAliveClientList){
                //检查客户端上次活跃事件,超过指定超时时间的就关闭
                Iterator<Client> clientIterator = serverConfigMeta.keepAliveClientList.iterator();
                while(clientIterator.hasNext()){
                    Client client = clientIterator.next();
                    if(client.socket.isClosed()){
                        logger.debug("客户端链接已经关闭,从缓存中移除!客户端:{}", client.socket);
                        clientIterator.remove();
                    }else{
                        logger.trace("检查超时时间.客户端上次活跃时间:{},套接字:{}",new Date(client.lastActiveTime),client.socket);
                        if(System.currentTimeMillis()-client.lastActiveTime>=serverConfigMeta.keepAliveTimeout){
                            logger.debug("关闭不活跃客户端,套接字:{}",new Date(client.lastActiveTime),client.socket);
                            clientIterator.remove();
                            try {
                                client.socket.close();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            }
        }, 3000, serverConfigMeta.keepAliveTimeout/2, TimeUnit.MILLISECONDS);
    }
}
