package cn.schoolwow.quickserver.response;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;

public class SpeedLimitOutputStream extends OutputStream {
    private Logger logger = LoggerFactory.getLogger(SpeedLimitOutputStream.class);

    /**
     * 最大速率1s内写入的字节数
     */
    private volatile long bytesPerSecond;
    /**
     * 记录总共接收字节大小
     */
    private volatile long totalSendBytes;
    /**
     * 记录上次接收字节大小
     */
    private volatile long lastSendBytes;
    /**
     * 上次记录时间
     */
    private volatile long lastRecordTime = System.nanoTime();
    /**
     * 包装的输入流
     */
    private OutputStream outputStream;

    public SpeedLimitOutputStream(OutputStream outputStream) {
        this.outputStream = outputStream;
    }

    /**
     * 设置最大下载速率(kb/s)
     */
    public synchronized void setMaxDownloadSpeed(int setMaxDownloadSpeed) {
        bytesPerSecond = setMaxDownloadSpeed * 1024;
    }

    /**
     * 限速下载操作
     */
    public synchronized void limit(int length) {
        totalSendBytes += length;
        while (bytesPerSecond > 0 && totalSendBytes - lastSendBytes > bytesPerSecond) {
            long waitNanoTime = 1000000000L - (System.nanoTime() - lastRecordTime);
            if (waitNanoTime > 0) {
                try {
                    Thread.sleep(waitNanoTime / 1000000, (int) (waitNanoTime % 1000000));
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            lastRecordTime = System.nanoTime();
            lastSendBytes += bytesPerSecond;
        }
    }

    @Override
    public void write(int b) throws IOException {
        limit(1);
        outputStream.write(b);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        limit(len);
        outputStream.write(b, off, len);
    }

    @Override
    public void write(byte[] b) throws IOException {
        if (b.length > 0) {
            limit(b.length);
        }
        outputStream.write(b);
    }
}