package cn.schoolwow.quickserver.response;

import cn.schoolwow.quickserver.domain.Client;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpCookie;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

public class HttpResponseImpl implements HttpResponse {
    private Logger logger = LoggerFactory.getLogger(HttpResponseImpl.class);

    /**客户端信息*/
    private Client client;

    public HttpResponseImpl(Client client) {
        this.client = client;
    }

    @Override
    public void status(HttpStatus httpStatus) {
        client.httpResponse.httpStatus(httpStatus);
    }

    @Override
    public void charset(String charset) {
        client.httpResponseMeta.charset = charset;
    }

    @Override
    public void addCookie(HttpCookie httpCookie) {
        StringBuilder builder = new StringBuilder(httpCookie.getName() + "=" + httpCookie.getValue() + ";");
        if (httpCookie.getMaxAge() > 0) {
            //expire为版本0,max-age为版本1
            String expires = LocalDateTime.now().plusSeconds(httpCookie.getMaxAge()).format(DateTimeFormatter.ofPattern("EEE, d MMM yyyy HH:mm:ss 'GMT'", Locale.US));
            builder.append(" expires=" + expires + ";");
        }
        if (null != httpCookie.getDomain()) {
            builder.append(" domain=" + httpCookie.getDomain() + ";");
        }
        if (null != httpCookie.getPath()) {
            builder.append(" path=" + httpCookie.getPath() + ";");
        }
        //TODO 以下注释内容加上会导致CookieManage报InvalidCookie警告,是判断没有=号造成的
//        if (httpCookie.getSecure()) {
//            builder.append(" secure;");
//        }
//        if (httpCookie.isHttpOnly()) {
//            builder.append(" httpOnly;");
//        }
        if (!client.httpResponseMeta.headers.containsKey("Set-Cookie")) {
            client.httpResponseMeta.headers.put("Set-Cookie", new ArrayList<>());
        }
        client.httpResponseMeta.headers.get("Set-Cookie").add(builder.toString());
    }

    @Override
    public List<String> getHeader(String name) {
        return client.httpResponseMeta.headers.get(name);
    }

    @Override
    public void addHeader(String name, String value) {
        if (!client.httpResponseMeta.headers.containsKey(name)) {
            client.httpResponseMeta.headers.put(name, new ArrayList<>());
        }
        client.httpResponseMeta.headers.get(name).add(value);
    }

    @Override
    public void setHeader(String name, String value) {
        client.httpResponseMeta.headers.put(name, new ArrayList<>(Arrays.asList(value)));
    }

    @Override
    public String getCharset() {
        return client.httpResponseMeta.charset;
    }

    @Override
    public void setCharset(String charset) {
        client.httpResponseMeta.charset = charset;
    }

    @Override
    public void setCharset(Charset charset) {
        client.httpResponseMeta.charset = charset.name();
    }

    @Override
    public void setContentLength(long contentLength) {
        client.httpResponseMeta.contentLength = contentLength;
    }

    @Override
    public String getContentType() {
        return client.httpResponseMeta.contentType;
    }

    @Override
    public void setContentType(String contentType) {
        client.httpResponseMeta.contentType = contentType;
    }

    @Override
    public String getContentDisposition() {
        List<String> values = getHeader("Content-Disposition");
        return values == null || values.size() == 0 ? null : values.get(0);
    }

    @Override
    public void setContentDisposition(String fileName) {
        try {
            String contentDisposition = "attachment;filename*=utf-8''" + new String(fileName.getBytes(), client.httpResponseMeta.charset);
            setHeader("Content-Disposition", contentDisposition);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void acceptRanges(boolean acceptRanges) {
        if (acceptRanges) {
            setHeader("Accept-Ranges", "bytes");
        } else {
            client.httpResponseMeta.headers.remove("Accept-Ranges");
        }
    }

    @Override
    public void redirect(String url) {
        client.httpResponse.httpStatus(HttpStatus.FOUND);
        client.httpResponseMeta.headers.put("Location", Arrays.asList(url));
    }

    @Override
    public void streamingBody(String data) throws IOException {
        if (null == client.httpResponseMeta.headerString) {
            setContentType("text/plain");
            setCharset(client.httpResponseMeta.charset);
            setHeader("Access-Control-Allow-Origin", "*");
            client.httpResponse.fillHeadStream();
        }
        client.httpResponseMeta.outputStream.write(data.getBytes(client.httpResponseMeta.charset));
        client.httpResponseMeta.outputStream.flush();
    }

    @Override
    public void streamingBodyLine(String data) throws IOException {
        streamingBody(data+"\n");
    }

    @Override
    public void eventSource(EventSource eventSource) throws IOException {
        if (null == client.httpResponseMeta.headerString) {
            setContentType("text/event-stream");
            setCharset("utf-8");
            client.httpResponse.fillHeadStream();
        }
        String body = "id:" + eventSource.id + "\r\nevent:" + eventSource.event + "\r\nretry:" + eventSource.retry + "\r\ndata:" + eventSource.data + "\r\n\r\n";
        client.httpResponseMeta.outputStream.write(body.getBytes(client.httpResponseMeta.charset));
        client.httpResponseMeta.outputStream.flush();
    }

    @Override
    public void download(String filePath) throws IOException {
        Path path = Paths.get(filePath);
        if(Files.notExists(path)){
            throw new IOException("文件不存在!"+filePath);
        }
        setContentLength(Files.size(path));
        setCharset("utf-8");
        setContentDisposition(path.toFile().getName());
        OutputStream outputStream = getOutputStream();
        byte[] content = Files.readAllBytes(path);
        outputStream.write(content);
        outputStream.flush();
    }

    @Override
    public OutputStream getOutputStream() throws IOException {
        if (null == client.httpResponseMeta.headerString) {
            client.httpResponse.fillHeadStream();
        }
        return client.httpResponseMeta.outputStream;
    }

    /**
     * 设置返回状态码
     */
    public void httpStatus(HttpStatus httpStatus) {
        client.httpResponseMeta.status = httpStatus.status;
        client.httpResponseMeta.statusMessage = httpStatus.statusMessage;
    }

    /**
     * 填充http头部
     */
    public void fillHeadStream() throws IOException {
        if (client.httpResponseMeta.status == 0) {
            client.httpResponse.httpStatus(HttpStatus.OK);
        }
        client.httpResponseMeta.headers.put("Content-Type", Arrays.asList(client.httpResponseMeta.contentType));
        if (client.httpResponseMeta.contentLength > 0) {
            client.httpResponseMeta.headers.put("Content-Length", Arrays.asList(client.httpResponseMeta.contentLength + ""));
        }
        if (null == client.httpResponseMeta.headerString) {
            StringBuilder builder = new StringBuilder();
            builder.append(client.httpResponseMeta.protocol + " " + client.httpResponseMeta.status + " " + client.httpResponseMeta.statusMessage + "\r\n");
            Set<Map.Entry<String, List<String>>> entrySet = client.httpResponseMeta.headers.entrySet();
            for (Map.Entry<String, List<String>> entry : entrySet) {
                for (String value : entry.getValue()) {
                    builder.append(entry.getKey() + ": " + value + "\r\n");
                }
            }
            builder.append("\r\n");
            client.httpResponseMeta.headerString = builder.toString();
            client.httpResponseMeta.outputStream.write(client.httpResponseMeta.headerString.getBytes(client.httpResponseMeta.charset));
            client.httpResponseMeta.outputStream.flush();
        } else {
            logger.warn("头部报文已写入");
        }
    }
}
