package cn.schoolwow.quickserver.handler;

import cn.schoolwow.quickserver.controller.RequestMethod;
import cn.schoolwow.quickserver.domain.Client;
import cn.schoolwow.quickserver.domain.ControllerMeta;
import cn.schoolwow.quickserver.interceptor.HandlerInterceptor;
import cn.schoolwow.quickserver.interceptor.Interceptor;
import cn.schoolwow.quickserver.util.AntPathMatcherUtil;
import cn.schoolwow.quickserver.util.QuickServerUtil;
import cn.schoolwow.quickserver.websocket.WebSocketServerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * 请求分发
 */
public class DispatcherHandler implements Handler {
    private Logger logger = LoggerFactory.getLogger(DispatcherHandler.class);

    @Override
    public Handler handle(Client client) throws Exception {
        Collection<HandlerInterceptor> interceptorList = client.serverConfigMeta.applicationContext.getBeansOfType(HandlerInterceptor.class).values();
        for (HandlerInterceptor handlerInterceptor : interceptorList) {
            if (matchInterceptor(client, handlerInterceptor)) {
                if(!handlerInterceptor.preHandle(client.httpRequest, client.httpResponse, client.httpSession)){
                    return new HttpResponseHandler();
                }
            }
        }

        getControllerMeta(client);
        getWebsocketServerListener(client);
        Handler handler = null;
        //根据请求类型判断是控制器,WebSocket还是静态资源
        if(null!=client.controllerMeta){
            logger.trace("处理方式为Controller,处理方法:{}", client.controllerMeta.method.toString());
            handler = new AnnotationHandler();
        }else if(null!=client.webSocketServerListener){
            logger.trace("处理方式为WebSocket,处理方法:{}", client.webSocketServerListener.getClass().getName());
            handler = new WebSocketHandler();
        }else{
            logger.trace("处理方式为静态资源");
            handler = new StaticResourceHandler();
        }
        while (null!=handler&&!(handler instanceof HttpResponseHandler)) {
            try {
                handler = handler.handle(client);
            }catch (Exception e){
                QuickServerUtil.handleExceptionHandler(client,e);
                handler = new HttpResponseHandler();
            }
        }

        for (HandlerInterceptor handlerInterceptor : interceptorList) {
            if (matchInterceptor(client, handlerInterceptor)) {
                handlerInterceptor.postHandle(client.httpRequest, client.httpResponse, client.httpSession);
            }
        }

        for (HandlerInterceptor handlerInterceptor : interceptorList) {
            if (matchInterceptor(client, handlerInterceptor)) {
                handlerInterceptor.beforeResponse(client.httpRequest, client.httpResponse, client.httpSession);
            }
        }
        return handler;
    }

    /**
     * 查找Controller
     */
    private void getControllerMeta(Client client) {
        for (ControllerMeta controllerMeta : client.serverConfigMeta.controllerMetaList) {
            int urlPos = 0, mappingPos = 0, lastUrlPos = 0, lastMappingPos = 0;
            String requestPath = client.httpRequestMeta.uri.getPath();
            String mappingUrl = controllerMeta.mappingUrl;
            String antRequestUrl = mappingUrl;
            while (urlPos < requestPath.length() && mappingPos < mappingUrl.length()) {
                if (mappingUrl.charAt(mappingPos) == '{') {
                    lastUrlPos = urlPos;
                    lastMappingPos = mappingPos + 1;

                    while (mappingPos < mappingUrl.length() && mappingUrl.charAt(mappingPos) != '}') {
                        mappingPos++;
                    }
                    if (mappingPos < mappingUrl.length()) {
                        //提取变量名
                        String name = mappingUrl.substring(lastMappingPos, mappingPos);
                        antRequestUrl = antRequestUrl.replace("{" + name + "}", "*");
                        String value = null;
                        //提取变量值
                        if (mappingPos + 1 < mappingUrl.length()) {
                            while (urlPos < requestPath.length() && requestPath.charAt(urlPos) != mappingUrl.charAt(mappingPos + 1)) {
                                urlPos++;
                            }
                            if (urlPos < requestPath.length()) {
                                value = requestPath.substring(lastUrlPos, urlPos);
                            }
                        } else {
                            value = requestPath.substring(lastUrlPos);
                        }
                        client.httpRequestMeta.pathVariable.put(name, value);
                    }
                } else if (requestPath.charAt(urlPos) == mappingUrl.charAt(mappingPos)) {
                    urlPos++;
                    mappingPos++;
                } else {
                    mappingPos++;
                }
            }
            if ((mappingUrl.contains("{") && AntPathMatcherUtil.match(requestPath, antRequestUrl))
                    || requestPath.equals(mappingUrl)) {
                //判断请求方法是否匹配
                for (RequestMethod requestMethod : controllerMeta.requestMethodList) {
                    if (requestMethod.name().equalsIgnoreCase(client.httpRequestMeta.method)) {
                        logger.trace("匹配Controller,路径:{},处理方法:{}", client.httpRequestMeta.uri.getPath(), controllerMeta.method.toString());
                        client.controllerMeta = controllerMeta;
                        return;
                    }
                }
            }
        }
    }

    /**
     * 查找WebSocket路径映射
     */
    private void getWebsocketServerListener(Client client) {
        Set<Map.Entry<String,WebSocketServerListener>> entrySet = client.serverConfigMeta.uriWebSocketServerListenerMap.entrySet();
        for(Map.Entry<String,WebSocketServerListener> entry:entrySet){
            if(entry.getKey().equalsIgnoreCase(client.httpRequestMeta.uri.getPath())){
                logger.trace("匹配WebSocket监听器,类名:{}", entry.getValue().getClass().getName());
                client.webSocketServerListener = entry.getValue();
                break;
            }
        }
    }

    /**
     * 路径是否匹配拦截器
     */
    private boolean matchInterceptor(Client client, HandlerInterceptor handlerInterceptor) {
        String uri = client.httpRequestMeta.uri.getPath();
        Interceptor interceptor = handlerInterceptor.getClass().getDeclaredAnnotation(Interceptor.class);
        if (null == interceptor) {
            return false;
        }
        for (String excludePattern : interceptor.excludePatterns()) {
            if (AntPathMatcherUtil.match(uri, excludePattern)) {
                return false;
            }
        }
        if(interceptor.excludeStaticResource()&&null==client.controllerMeta){
            return false;
        }
        for (String pattern : interceptor.patterns()) {
            if (AntPathMatcherUtil.match(uri, pattern)) {
                return true;
            }
        }
        return false;
    }
}