package cn.schoolwow.quickserver.handler;

import cn.schoolwow.quickserver.controller.annotation.*;
import cn.schoolwow.quickserver.domain.Client;
import cn.schoolwow.quickserver.domain.MultipartFile;
import cn.schoolwow.quickserver.request.HttpRequest;
import cn.schoolwow.quickserver.response.HttpResponse;
import cn.schoolwow.quickserver.response.HttpStatus;
import cn.schoolwow.quickserver.session.HttpSession;
import cn.schoolwow.quickserver.util.QuickServerUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Parameter;
import java.net.HttpCookie;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 控制器处理
 */
public class ControllerHandler implements Handler {
    private Logger logger = LoggerFactory.getLogger(ControllerHandler.class);

    @Override
    public Handler handle(Client client) throws Exception {
        Object[] parameterArray = getMethodParameter(client);
        logger.debug("调用Controller,类名:{},方法名:{},参数列表:{}",
                client.controllerMeta.method.getDeclaringClass().getName(),
                client.controllerMeta.method.getName(),
                Arrays.toString(parameterArray)
        );
        Object result = client.controllerMeta.method.invoke(client.controllerMeta.instance, parameterArray);
        if (null != client.serverConfigMeta.responseBodyAdvice && client.serverConfigMeta.responseBodyAdvice.support(client.controllerMeta.method)) {
            result = client.serverConfigMeta.responseBodyAdvice.beforeBodyWrite(result, client.controllerMeta.method, client.httpRequest, client.httpResponse, client.httpSession);
        }
        handleResult(result, client);
        return new HttpResponseHandler();
    }

    /**获取方法参数列表*/
    private Object[] getMethodParameter(Client client) throws Exception {
        Object[] parameterArray = new Object[client.controllerMeta.method.getParameterCount()];
        Parameter[] parameters = client.controllerMeta.method.getParameters();
        for (int i = 0; i < parameters.length; i++) {
            Parameter parameter = parameters[i];
            Object parameterValue = getAutoWireType(client, parameter);
            if (null == parameterValue) {
                parameterValue = getPathVariable(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getRequestParam(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getRequestPart(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getRequestBody(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getSessionValue(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getRequestHeader(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getCookieValue(client, parameter);
            }
            if (null == parameterValue) {
                parameterValue = getCompositeParameter(client, parameter);
            }

            if (null != parameterValue) {
                //参数类型转换
                parameterArray[i] = TypeUtils.cast(parameterValue, parameter.getType(), null);
            }
        }
        return parameterArray;
    }

    /**
     * 获取自动装配类型
     */
    private Object getAutoWireType(Client client, Parameter parameter) {
        if (parameter.getType().getName().equals(HttpRequest.class.getName())) {
            return client.httpRequest;
        } else if (parameter.getType().getName().equals(HttpResponse.class.getName())) {
            return client.httpResponse;
        } else if (parameter.getType().getName().equals(HttpSession.class.getName())) {
            return client.httpSession;
        }
        return null;
    }

    /**
     * 处理PathVariable注解
     */
    private Object getPathVariable(Client client, Parameter parameter) {
        PathVariable pathVariable = parameter.getAnnotation(PathVariable.class);
        if (null == pathVariable) {
            return null;
        }
        if (pathVariable.required() && !client.httpRequestMeta.pathVariable.containsKey(pathVariable.name())) {
            throw new IllegalArgumentException("路径变量[" + pathVariable.name() + "]不能为空!");
        }
        return client.httpRequestMeta.pathVariable.get(pathVariable.name());
    }

    /**
     * 处理RequestParam注解
     */
    private Object getRequestParam(Client client, Parameter parameter) throws ParseException {
        RequestParam requestParam = parameter.getAnnotation(RequestParam.class);
        if (null == requestParam) {
            return null;
        }
        String requestParameter = client.httpRequestMeta.dataMap.get(requestParam.name());
        if (requestParam.required() && requestParameter == null) {
            throw new IllegalArgumentException("表单请求参数[" + requestParam.name() + "]不能为空!");
        }
        if (requestParameter == null || requestParameter.isEmpty()) {
            requestParameter = requestParam.defaultValue();
        }
        //处理日期类型
        String pattern = requestParam.pattern();
        if (requestParam.required() && !pattern.isEmpty() && requestParameter.isEmpty()) {
            throw new IllegalArgumentException("表单请求参数[" + requestParam.name() + "]不能为空!");
        }
        if (!pattern.isEmpty() && !requestParameter.isEmpty()) {
            return getDateParameterByType(pattern, parameter.getType(), requestParameter);
        }
        return requestParameter;
    }

    /**
     * 处理RequestPart注解
     */
    private Object getRequestPart(Client client, Parameter parameter) throws ParseException {
        RequestPart requestPart = parameter.getAnnotation(RequestPart.class);
        if (null == requestPart) {
            return null;
        }
        switch (parameter.getType().getName()) {
            case "cn.schoolwow.quickserver.domain.MultipartFile": {
                List<MultipartFile> multipartFileList = client.httpRequestMeta.fileParameterMap.get(requestPart.name());
                if (requestPart.required() && (multipartFileList == null || multipartFileList.isEmpty())) {
                    throw new IllegalArgumentException("表单请求参数[" + requestPart.name() + "]不能为空!");
                }
                if (null == multipartFileList || multipartFileList.isEmpty()) {
                    return null;
                }
                return multipartFileList.get(0);
            }
            case "[Lcn.schoolwow.quickserver.domain.MultipartFile;": {
                List<MultipartFile> multipartFileList = client.httpRequestMeta.fileParameterMap.get(requestPart.name());
                if (requestPart.required() && (multipartFileList == null || multipartFileList.isEmpty())) {
                    throw new IllegalArgumentException("表单请求参数[" + requestPart.name() + "]不能为空!");
                }
                if (null == multipartFileList || multipartFileList.isEmpty()) {
                    return new MultipartFile[0];
                }
                return multipartFileList;
            }
            default: {
                if (requestPart.required() && !client.httpRequestMeta.dataMap.containsKey(requestPart.name())) {
                    throw new IllegalArgumentException("表单请求参数[" + requestPart.name() + "]不能为空!");
                }
                String requestParameter = client.httpRequestMeta.dataMap.get(requestPart.name());
                if (requestParameter == null || requestParameter.isEmpty()) {
                    requestParameter = requestPart.defaultValue();
                }
                //处理日期类型
                String pattern = requestPart.pattern();
                if (requestPart.required() && !pattern.isEmpty() && requestParameter.isEmpty()) {
                    throw new IllegalArgumentException("表单请求参数[" + requestPart.name() + "]不能为空!");
                }
                if (!pattern.isEmpty() && !requestParameter.isEmpty()) {
                    return getDateParameterByType(pattern, parameter.getType(), requestParameter);
                }
                return requestParameter;
            }
        }
    }

    /**
     * 处理RequestBody注解
     */
    private Object getRequestBody(Client client, Parameter parameter) {
        RequestBody requestBody = parameter.getAnnotation(RequestBody.class);
        if (null == requestBody) {
            return null;
        }
        Class type = parameter.getType();
        if (isPrimitiveType(type)) {
            return TypeUtils.cast(client.httpRequestMeta.body, parameter.getType(), null);
        } else if (type.getName().equals(String.class.getName())) {
            return client.httpRequestMeta.body;
        } else if (client.httpRequestMeta.body.startsWith("{")) {
            if(parameter.getType().getName().equals(JSONObject.class.getName())){
                return JSON.parseObject(client.httpRequestMeta.body);
            }
            return JSON.parseObject(client.httpRequestMeta.body).toJavaObject(parameter.getType());
        } else if (client.httpRequestMeta.body.startsWith("[")) {
            if(parameter.getType().getName().equals(JSONArray.class.getName())){
                return JSON.parseArray(client.httpRequestMeta.body);
            }
            return JSON.parseArray(client.httpRequestMeta.body).toJavaList(parameter.getType().getComponentType());
        } else if (requestBody.required()) {
            throw new IllegalArgumentException("@RequestBody注解无法解析该JSON参数!参数名称:" + parameter.getName() + ",参数值:" + client.httpRequestMeta.body);
        }
        return null;
    }

    /**
     * 处理SessionValue注解
     */
    private Object getSessionValue(Client client, Parameter parameter) {
        SessionValue sessionValue = parameter.getAnnotation(SessionValue.class);
        if (null == sessionValue) {
            return null;
        }
        if (sessionValue.required() && !client.httpSessionMeta.attributes.containsKey(sessionValue.name())) {
            throw new IllegalArgumentException("SessionValue注解[" + sessionValue.name() + "]不能为空!");
        }
        Object parameterValue = client.httpSessionMeta.attributes.get(sessionValue.name());
        logger.trace("[SessionValue]name:{},value:{},defaultValue:{}", sessionValue.name(), parameterValue, sessionValue.defaultValue());
        if (null == parameterValue ) {
            parameterValue = sessionValue.defaultValue();
        }
        return parameterValue;
    }

    /**
     * 处理RequestHeader注解
     */
    private Object getRequestHeader(Client client, Parameter parameter) {
        RequestHeader requestHeader = parameter.getAnnotation(RequestHeader.class);
        if (null == requestHeader) {
            return null;
        }
        List<String> requestHeaderValues = client.httpRequestMeta.headers.get(requestHeader.name());
        if (requestHeader.required() && null == requestHeaderValues) {
            throw new IllegalArgumentException("http头部[" + requestHeader.name() + "]不能为空!");
        }
        Object parameterValue = null == requestHeaderValues ? null : requestHeaderValues.get(0);
        logger.trace("[RequestHeader]name:{},value:{},defaultValue:{}", requestHeader.name(), parameterValue, requestHeader.defaultValue());
        if (null == parameterValue) {
            parameterValue = requestHeader.defaultValue();
        }
        return parameterValue;
    }

    /**
     * 处理CookieValue注解
     */
    private Object getCookieValue(Client client, Parameter parameter) {
        CookieValue cookieValue = parameter.getAnnotation(CookieValue.class);
        if (null == cookieValue) {
            return null;
        }
        HttpCookie httpCookie = null;
        for (HttpCookie _httpCookie : client.httpRequestMeta.cookieList) {
            if (_httpCookie.getName().equals(cookieValue.name())) {
                httpCookie = _httpCookie;
                break;
            }
        }
        if (null == httpCookie) {
            throw new IllegalArgumentException("Cookie[" + cookieValue.name() + "]不存在!");
        }
        return httpCookie.getValue();
    }

    /**获取复杂参数处理参数*/
    private Object getCompositeParameter(Client client, Parameter parameter) throws Exception {
        Class parameterType = parameter.getType();
        if (isPrimitiveType(parameterType) || "java.lang.String".equals(parameterType.getName())) {
            return null;
        }
        Object instance = parameterType.newInstance();
        Field[] fields = parameterType.getDeclaredFields();
        Field.setAccessible(fields, true);
        for (Field field : fields) {
            if (client.httpRequestMeta.dataMap.containsKey(field.getName())) {
                field.set(instance, client.httpRequestMeta.dataMap.get(field.getName()));
            }
        }
        return instance;
    }

    /**
     * 处理返回结果
     */
    private void handleResult(Object result, Client client) throws IOException {
        if(null==result){
            return;
        }
        //字节数组
        if(result instanceof byte[]){
            if(null==client.httpResponseMeta.contentType){
                client.httpResponseMeta.contentType = "text/plain";
            }
            byte[] bytes = (byte[]) result;
            client.httpResponseMeta.result = QuickServerUtil.byteArrayToHex(bytes);
            client.httpResponseMeta.contentLength = bytes.length;
            client.httpResponseMeta.bodyInputStream = new ByteArrayInputStream(bytes);
            return;
        }
        String resultString = result.toString();
        //重定向
        if (resultString.startsWith("redirect:")) {
            client.httpResponse.httpStatus(HttpStatus.FOUND);
            client.httpResponseMeta.headers.put("Location", Arrays.asList(resultString.substring("redirect:".length())));
            return;
        }
        //基本数据类型或者字符串
        if (isPrimitiveType(result.getClass()) || result instanceof String) {
            if(null==client.httpResponseMeta.contentType){
                client.httpResponseMeta.contentType = "text/plain";
            }
            client.httpResponseMeta.result = String.valueOf(result);
            byte[] bytes = client.httpResponseMeta.result.getBytes(client.httpResponseMeta.charset);
            client.httpResponseMeta.contentLength = bytes.length;
            client.httpResponseMeta.bodyInputStream = new ByteArrayInputStream(bytes);
            return;
        }
        //处理成JSON字符串
        if(null==client.httpResponseMeta.contentType){
            client.httpResponseMeta.contentType = "application/json";
        }
        client.httpResponseMeta.result = JSON.toJSONString(result,client.serverConfigMeta.serializerFeatures);
        byte[] bytes = client.httpResponseMeta.result.getBytes(client.httpResponseMeta.charset);
        client.httpResponseMeta.contentLength = bytes.length;
        client.httpResponseMeta.bodyInputStream = new ByteArrayInputStream(bytes);
    }

    /**
     * 根据pattern和参数类型转换日期类型参数
     *
     * @param pattern          日期格式
     * @param parameterType    参数类型
     * @param requestParameter 请求参数
     */
    private Object getDateParameterByType(String pattern, Class parameterType, String requestParameter) throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        Date date = sdf.parse(requestParameter);
        switch (parameterType.getName()) {
            case "java.util.Date": {
                return date;
            }
            case "java.sql.Date": {
                return new java.sql.Date(date.getTime());
            }
            case "java.sql.Timestamp": {
                return new Timestamp(date.getTime());
            }
            case "java.util.Calendar": {
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(date);
                return calendar;
            }
            case "java.time.LocalDate": {
                return LocalDate.parse(requestParameter, DateTimeFormatter.ofPattern(pattern));
            }
            case "java.time.LocalDateTime": {
                return LocalDateTime.parse(requestParameter, DateTimeFormatter.ofPattern(pattern));
            }
            default: {
                throw new IllegalArgumentException("不支持该日期类型,目前支持的类型为Date,Calendar,LocalDate,LocalDateTime!当前类型:" + parameterType.getName());
            }
        }
    }

    /**
     * 是否为基本数据类型
     *
     * @param clazz 类型
     */
    private boolean isPrimitiveType(Class<?> clazz) {
        try {
            if (clazz.isPrimitive()) {
                return true;
            }
            return ((Class<?>) clazz.getField("TYPE").get(null)).isPrimitive();
        } catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
            return false;
        }
    }
}
