package cn.schoolwow.quickserver;

import cn.schoolwow.quickserver.domain.ServerConfigMeta;
import cn.schoolwow.quickserver.server.HttpServer;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

public class QuickServer {
    private Logger logger = LoggerFactory.getLogger(QuickServer.class);
    private ServerConfigMeta serverConfigMeta = new ServerConfigMeta();

    /**
     * 新建QuickServer实例
     */
    public static QuickServer newInstance() {
        JSON.DEFFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
        return new QuickServer();
    }

    /**
     * 获取配置元信息
     */
    public ServerConfigMeta serverConfigMeta() {
        return serverConfigMeta;
    }

    /**
     * 指定端口
     *
     * @param port 端口,为0表示使用系统分配端口
     */
    public QuickServer port(int port) {
        serverConfigMeta.port = port;
        return this;
    }

    /**
     * 注册指定类
     *
     * @param componentClass 容器类
     */
    public QuickServer register(Class... componentClasses) {
        serverConfigMeta.applicationContext.register(componentClasses);
        return this;
    }

    /**
     * 扫描指定包
     *
     * @param basePackages 指定包名
     */
    public QuickServer scan(String... basePackages) {
        serverConfigMeta.applicationContext.scan(basePackages);
        return this;
    }

    /**
     * 指定静态路径前缀
     *
     * @param staticResourcePathPrefix 静态路径前缀
     */
    public QuickServer staticResourcePathPrefix(String staticResourcePathPrefix) {
        logger.info("[指定静态路径前缀]{}",staticResourcePathPrefix);
        if(!staticResourcePathPrefix.startsWith("/")){
            staticResourcePathPrefix = "/" + staticResourcePathPrefix;
        }
        serverConfigMeta.staticResourcePathPrefix = staticResourcePathPrefix;
        return this;
    }

    /**
     * 指定静态路径
     *
     * @param staticResourcePath 静态路径
     */
    public QuickServer staticResourcePath(String staticResourcePath) {
        logger.info("[指定静态路径]{}",staticResourcePath);
        serverConfigMeta.staticResourcePathMap.put(staticResourcePath,"");
        return this;
    }

    /**
     * 指定静态路径
     *
     * @param staticResourcePath 静态路径
     */
    public QuickServer staticResourcePath(String staticResourcePath, String urlPrefix) {
        logger.info("[指定静态路径]路径:{},url前缀:{}",staticResourcePath,urlPrefix);
        serverConfigMeta.staticResourcePathMap.put(staticResourcePath,urlPrefix);
        return this;
    }

    /**
     * 是否开启keepAlive特性
     *
     * @param keepAlive 是否开启keepAlive特性
     */
    public QuickServer keepAlive(boolean keepAlive) {
        serverConfigMeta.keepAlive = keepAlive;
        return this;
    }

    /**
     * 指定keepAlive超时时间(毫秒)
     *
     * @param keepAliveTimeout keepAlive超时时间
     */
    public QuickServer keepAliveTimeout(int keepAliveTimeout) {
        serverConfigMeta.keepAliveTimeout = keepAliveTimeout;
        return this;
    }

    /**
     * 指定最大下载速率(kb/s)
     *
     * @param maxLimitSpeed 最大下载速率(kb/s)
     */
    public QuickServer maxLimitSpeed(int maxLimitSpeed) {
        serverConfigMeta.maxLimitSpeed = maxLimitSpeed;
        return this;
    }

    /**
     * 指定JSON序列化特性
     *
     * @param serializerFeatures JSON序列化特性
     */
    public QuickServer serializerFeature(SerializerFeature... serializerFeatures) {
        serverConfigMeta.serializerFeatures = serializerFeatures;
        return this;
    }

    /**
     * 指定JSON序列化特性
     *
     * @param serializerFeatures JSON序列化特性
     */
    public QuickServer serializerDateFormat(String dateTimeFormat) {
        JSON.DEFFAULT_DATE_FORMAT = dateTimeFormat;
        return this;
    }

    public void start() throws IOException {
        HttpServer httpServer = new HttpServer(serverConfigMeta);
        httpServer.start();
    }
}