package cn.schoolwow.quickhttp.module.webSocket.service;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickhttp.domain.websocket.WebSocketState;
import cn.schoolwow.quickhttp.module.common.HttpClientOption;
import cn.schoolwow.quickhttp.module.webSocket.domain.WebSocketClientConfig;
import cn.schoolwow.quickhttp.module.webSocket.domain.WebSocketStream;
import cn.schoolwow.quickhttp.module.webSocket.domain.WebSocketStreamImpl;
import cn.schoolwow.quickhttp.util.WebSocketUtil;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.*;

public class InitialWebSocketServiceFlow implements BusinessFlow {
    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        flowContext.putContextTemporaryInstanceData(WebSocketState.Connecting);

        setHandShakeHeaderMap(flowContext);
        setHandShakeContent(flowContext);
        sendHandShakeRequest(flowContext);
        checkResponseLine(flowContext);
        checkResponseHeader(flowContext);

        flowContext.putContextTemporaryInstanceData(WebSocketState.Connected);

        Object readLock = new Object();
        flowContext.putContextTemporaryData("readLock", readLock);

        Object writeLock = new Object();
        flowContext.putContextTemporaryData("writeLock", writeLock);
    }

    @Override
    public String name() {
        return "初始化WebSocket链接服务流程";
    }

    private void setHandShakeHeaderMap(FlowContext flowContext){
        String url = flowContext.checkData("url", String.class);
        HttpClientOption httpClientOption = flowContext.checkInstanceData(HttpClientOption.class);

        URI uri = URI.create(url);
        byte[] randomBytes = new byte[16];
        WebSocketUtil.randomBytes(randomBytes);
        String secWebSocketKey = Base64.getEncoder().encodeToString(randomBytes);

        Map<String, List<String>> headerMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        headerMap.put("Host", new ArrayList<>(Arrays.asList(uri.getHost())));
        headerMap.put("Upgrade", new ArrayList<>(Arrays.asList("websocket")));
        headerMap.put("Connection", new ArrayList<>(Arrays.asList("Upgrade")));
        headerMap.put("Sec-WebSocket-Key", new ArrayList<>(Arrays.asList(secWebSocketKey)));
        headerMap.put("Sec-WebSocket-Version", new ArrayList<>(Arrays.asList("13")));

        //设置Cookie
        String cookieHeaderValue = httpClientOption.cookieOption.getCookieString(uri.getHost());
        headerMap.put("Cookie", new ArrayList<>(Arrays.asList(cookieHeaderValue)));

        flowContext.putCurrentFlowData("uri", uri);
        flowContext.putCurrentFlowData("headerMap", headerMap);
        flowContext.putCurrentFlowData("secWebSocketKey", secWebSocketKey);
    }

    private void setHandShakeContent(FlowContext flowContext){
        Map<String, List<String>> headerMap = flowContext.checkData("headerMap", Map.class);
        WebSocketClientConfig webSocketClientConfig = flowContext.checkInstanceData(WebSocketClientConfig.class);
        URI uri = (URI) flowContext.checkData("uri");

        if(!webSocketClientConfig.handShakeHeaderMap.isEmpty()){
            headerMap.putAll(webSocketClientConfig.handShakeHeaderMap);
        }
        StringBuilder builder = new StringBuilder("GET "+uri.getPath()+" HTTP/1.1\r\n");
        Set<Map.Entry<String,List<String>>> entrySet = headerMap.entrySet();
        for(Map.Entry<String,List<String>> entry:entrySet){
            for(String value:entry.getValue()){
                builder.append(entry.getKey()+": "+value+"\r\n");
            }
        }
        builder.append("\r\n");
        flowContext.putCurrentFlowData("handShakeContent", builder.toString());
    }

    private void sendHandShakeRequest(FlowContext flowContext) throws Exception{
        URI uri = flowContext.checkData("uri", URI.class);
        String handShakeContent = flowContext.checkData("handShakeContent", String.class);
        WebSocketClientConfig webSocketClientConfig = flowContext.checkInstanceData(WebSocketClientConfig.class);

        Socket socket = new Socket();
        socket.connect(new InetSocketAddress(uri.getHost(),uri.getPort()), webSocketClientConfig.connectTimeout);
        WebSocketStream webSocketStream = new WebSocketStreamImpl(socket);
        webSocketStream.write(handShakeContent.getBytes(StandardCharsets.UTF_8));
        webSocketStream.flush();

        flowContext.putContextTemporaryInstanceData(socket, Socket.class);
        flowContext.putContextTemporaryInstanceData(webSocketStream);
    }

    private void checkResponseLine(FlowContext flowContext) throws Exception{
        WebSocketStream webSocketStream = flowContext.checkInstanceData(WebSocketStream.class);

        String responseLine = webSocketStream.readLine();
        flowContext.putData("响应行", responseLine);
        String protocol = responseLine.substring(0, responseLine.indexOf(' '));
        if(!"HTTP/1.1".equalsIgnoreCase(protocol)){
            throw new IOException("当前仅支持HTTP/1.1!服务端版本:"+protocol);
        }
        int status = Integer.parseInt(responseLine.substring(protocol.length()+1, protocol.length()+4));
        if(101!=status){
            throw new IOException(responseLine.substring(Math.min(responseLine.length(),protocol.length()+5)));
        }
    }

    private void checkResponseHeader(FlowContext flowContext) throws Exception{
        WebSocketStream webSocketStream = flowContext.checkInstanceData(WebSocketStream.class);
        String secWebSocketKey = flowContext.checkData("secWebSocketKey", String.class);

        TreeMap<String, List<String>> headers = webSocketStream.readHeaders();
        flowContext.putData("响应头部", headers);
        if(!headers.containsKey("upgrade")||!"websocket".equalsIgnoreCase(headers.get("upgrade").get(0))){
            throw new IOException("响应头部未包含upgrade或者upgrade的值不为websocket!");
        }
        if(!headers.containsKey("connection")||!"upgrade".equalsIgnoreCase(headers.get("connection").get(0))){
            throw new IOException("响应头部未包含connection或者connection的值不为upgrade!");
        }
        if(!headers.containsKey("Sec-WebSocket-Accept")){
            throw new IOException("响应头部未包含Sec-WebSocket-Accept!");
        }
        String actualSecWebSocketAccept = headers.get("Sec-WebSocket-Accept").get(0);
        String expectSecWebSocketAccept = WebSocketUtil.calculateSecWebSocketAccept(secWebSocketKey);
        if(!expectSecWebSocketAccept.equalsIgnoreCase(actualSecWebSocketAccept)){
            throw new IOException( "响应头部Sec-WebSocket-Accept的值不正确!");
        }
//        if(!headers.containsKey("Sec-WebSocket-Version")||!"13".equalsIgnoreCase(headers.get("Sec-WebSocket-Version").get(0))){
//            throw new IOException("响应头部未包含Sec-WebSocket-Version或者Sec-WebSocket-Version的值不为13!");
//        }
    }
}
