package cn.schoolwow.quickhttp.module.webSocket.kit;

import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.QuickFlowBuilder;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickhttp.domain.websocket.OPCode;
import cn.schoolwow.quickhttp.domain.websocket.WebSocketClient;
import cn.schoolwow.quickhttp.domain.websocket.WebSocketFrame;
import cn.schoolwow.quickhttp.domain.websocket.WebSocketState;
import cn.schoolwow.quickhttp.module.webSocket.domain.WebSocketClientConfig;
import cn.schoolwow.quickhttp.module.webSocket.domain.WebSocketMode;
import cn.schoolwow.quickhttp.module.webSocket.service.*;

import java.net.Socket;
import java.nio.charset.StandardCharsets;

/**WebSocket客户端*/
public class WebSocketClientImpl implements WebSocketClient {
    private QuickFlow webSocketFlow;

    public WebSocketClientImpl(String url, WebSocketClientConfig webSocketClientConfig, QuickFlow httpFlow) {
        if(null==webSocketClientConfig){
            webSocketClientConfig = new WebSocketClientConfig();
        }
        webSocketFlow = QuickFlowBuilder.newInstance()
                .parentFlow(httpFlow);
        webSocketFlow.startFlow(new InitialWebSocketServiceFlow())
                .putTemporaryData("url", url)
                .putTemporaryData("webSocketClientConfig", webSocketClientConfig)
                .execute();
    }

    @Override
    public void sendText(String text) {
        WebSocketFrame webSocketFrame = WebSocketFrame.newInstance()
                .opCode(OPCode.TextFrame)
                .mask(true)
                .payload(text.getBytes(StandardCharsets.UTF_8));
        webSocketFlow.startFlow(new WriteWebSocketFrameFlow())
                .putTemporaryInstanceData(webSocketFrame)
                .execute();
    }

    @Override
    public void sendBinary(byte[] data) {
        WebSocketFrame webSocketFrame = WebSocketFrame.newInstance()
                .opCode(OPCode.BinaryFrame)
                .mask(true)
                .payload(data);
        webSocketFlow.startFlow(new WriteWebSocketFrameFlow())
                .putTemporaryInstanceData(webSocketFrame)
                .execute();
    }

    @Override
    public void sendWebSocketFrame(WebSocketFrame webSocketFrame) {
        webSocketFlow.startFlow(new WriteWebSocketFrameFlow())
                .putTemporaryInstanceData(webSocketFrame)
                .execute();
    }

    @Override
    public String receiveText() {
        FlowContext flowContext = webSocketFlow.startFlow(new ReceiveWebSocketTextServiceFlow())
                .putTemporaryInstanceData(WebSocketMode.Server)
                .putTemporaryData("expectCode", OPCode.TextFrame)
                .execute();
        String text = flowContext.getData("text", String.class);
        return text;
    }

    @Override
    public byte[] receiveBinary() {
        FlowContext flowContext = webSocketFlow.startFlow(new ReceiveWebSocketBinaryServiceFlow())
                .putTemporaryInstanceData(WebSocketMode.Server)
                .putTemporaryData("expectCode", OPCode.BinaryFrame)
                .execute();
        WebSocketFrame webSocketFrame = flowContext.checkInstanceData(WebSocketFrame.class);
        return webSocketFrame.payload;
    }

    @Override
    public synchronized void ping() {
        webSocketFlow.startFlow(new SendWebSocketPingFrameServiceFlow())
                .putTemporaryInstanceData(WebSocketMode.Server)
                .putTemporaryData("expectCode", OPCode.PongFrame)
                .execute();
    }

    @Override
    public WebSocketFrame receiveWebSocketFrame() {
        FlowContext flowContext = webSocketFlow.startFlow(new ReceiveWebSocketFrameServiceFlow())
                .putTemporaryInstanceData(WebSocketMode.Server)
                .execute();
        WebSocketFrame webSocketFrame = flowContext.checkInstanceData(WebSocketFrame.class);
        return webSocketFrame;
    }

    @Override
    public void close() {
        webSocketFlow.executeFlowList(new CloseWebSocketStreamServiceFlow());
    }

    @Override
    public WebSocketState getState() {
        WebSocketState webSocketState = webSocketFlow.getContextInstanceData(WebSocketState.class);
        return webSocketState;
    }

    @Override
    public Socket getSocket() {
        Socket socket = webSocketFlow.getContextInstanceData(Socket.class);
        return socket;
    }

}
