package cn.schoolwow.quickhttp.module.request.execute.flow.listener;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickhttp.domain.execute.HttpRequestOption;
import cn.schoolwow.quickhttp.domain.execute.HttpResponseContext;
import cn.schoolwow.quickhttp.domain.execute.Request;
import cn.schoolwow.quickhttp.module.common.DataFileContext;
import cn.schoolwow.quickhttp.module.common.HttpClientOption;
import cn.schoolwow.quickhttp.module.common.HttpRequestContext;

import java.io.IOException;
import java.net.HttpCookie;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.*;

public class BeforeExecuteHttpRequestFlow implements BusinessFlow {
    private static final char[] mimeBoundaryChars =
            "-_1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ".toCharArray();
    private static final Random rand = new Random();
    private static final int boundaryLength = 32;

    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        checkRequestContext(flowContext);
        executeBeforeExecuteListener(flowContext);
        setUrlParameter(flowContext);
        setCookieHeader(flowContext);
        setContentType(flowContext);
    }

    @Override
    public String name() {
        return "执行http请求之前流程";
    }

    private void checkRequestContext(FlowContext flowContext){
        HttpRequestOption httpRequestOption = flowContext.checkInstanceData(HttpRequestOption.class);
        HttpClientOption httpClientOption = flowContext.checkInstanceData(HttpClientOption.class);

        if (null == httpRequestOption.url) {
            flowContext.printFlowDataRange();
            System.out.println(flowContext.getQuickFlow().getParentFlow().getContextDataMap().keySet());
            throw new IllegalArgumentException("url不能为空");
        }
        String protocol = httpRequestOption.url.getProtocol();
        if (!protocol.startsWith("http")) {
            throw new IllegalArgumentException("当前只支持http和https协议!当前url:"+httpRequestOption.url);
        }

        if (null == httpRequestOption.proxy) {
            httpRequestOption.proxy = httpClientOption.proxy;
        }
        if (3000 == httpRequestOption.connectTimeoutMillis) {
            httpRequestOption.connectTimeoutMillis = httpClientOption.connectTimeoutMillis;
        }
        if (5000 == httpRequestOption.readTimeoutMillis) {
            httpRequestOption.readTimeoutMillis = httpClientOption.readTimeoutMillis;
        }
        if (httpRequestOption.followRedirects) {
            httpRequestOption.followRedirects = httpClientOption.followRedirects;
        }
        if (20 == httpRequestOption.maxFollowRedirectTimes) {
            httpRequestOption.maxFollowRedirectTimes = httpClientOption.maxFollowRedirectTimes;
        }
        if (!httpRequestOption.ignoreHttpErrors) {
            httpRequestOption.ignoreHttpErrors = httpClientOption.ignoreHttpErrors;
        }
        if (3 == httpRequestOption.retryTimes) {
            httpRequestOption.retryTimes = httpClientOption.retryTimes;
        }
    }

    private void executeBeforeExecuteListener(FlowContext flowContext) throws IOException {
        HttpClientOption httpClientOption = flowContext.checkInstanceData(HttpClientOption.class);
        Request request = flowContext.checkInstanceData(Request.class);

        if(null==httpClientOption.quickHttpClientListener){
            return;
        }
        if(!httpClientOption.quickHttpClientListener.beforeExecute(request)){
            flowContext.broken("因beforeExecute事件返回false而停止执行");
        }
    }

    private void setUrlParameter(FlowContext flowContext) throws Exception {
        HttpRequestOption httpRequestOption = flowContext.checkInstanceData(HttpRequestOption.class);

        if (httpRequestOption.parameterMap.isEmpty()) {
            return;
        }
        //添加路径请求参数
        StringBuilder parameterBuilder = new StringBuilder();
        Set<Map.Entry<String, Object>> entrySet = httpRequestOption.parameterMap.entrySet();
        for (Map.Entry<String, Object> entry : entrySet) {
            Object value = entry.getValue();
            if (null != value) {
                value = URLEncoder.encode(value.toString(), httpRequestOption.charset);
            }
            parameterBuilder.append(URLEncoder.encode(entry.getKey(), httpRequestOption.charset) + "=" + value + "&");
        }
        parameterBuilder.deleteCharAt(parameterBuilder.length() - 1);
        if (httpRequestOption.url.toString().contains("?")) {
            parameterBuilder.insert(0, "&");
        } else {
            parameterBuilder.insert(0, "?");
        }
        httpRequestOption.url = new URL(httpRequestOption.url + parameterBuilder.toString());
    }

    private void setCookieHeader(FlowContext flowContext){
        HttpRequestOption httpRequestOption = flowContext.checkInstanceData(HttpRequestOption.class);
        HttpClientOption httpClientOption = flowContext.checkInstanceData(HttpClientOption.class);

        StringBuilder cookieHeader = new StringBuilder();
        //检查匹配域名和路径即可
        for(HttpCookie httpCookie:httpClientOption.httpCookieList){
            if(httpCookie.hasExpired()){
                continue;
            }
            if(null!=httpCookie.getDomain()&&!httpRequestOption.url.getHost().startsWith(httpCookie.getDomain())){
                continue;
            }
            if(null!=httpCookie.getPath()&&!httpRequestOption.url.getPath().startsWith(httpCookie.getPath())){
                continue;
            }
            cookieHeader.append(httpCookie.getName()+"="+httpCookie.getValue()+"; ");
        }
        if(cookieHeader.length()>0){
            httpRequestOption.headerMap.put("Cookie", Arrays.asList(cookieHeader.toString()));
        }
    }

    private void setContentType(FlowContext flowContext) throws Exception {
        HttpRequestOption httpRequestOption = flowContext.checkInstanceData(HttpRequestOption.class);
        HttpRequestContext httpRequestContext = flowContext.checkInstanceData(HttpRequestContext.class);

        if(null==httpRequestOption.contentType){
            if(null!=httpRequestOption.userContentType){
                httpRequestOption.contentType = httpRequestOption.userContentType.value;
            }else if(!httpRequestOption.dataFileMap.isEmpty()){
                httpRequestOption.contentType = Request.ContentType.MULTIPART_FORMDATA.value;
            }else if((httpRequestOption.requestBody != null && httpRequestOption.requestBody.length > 0)){
                httpRequestOption.contentType = Request.ContentType.APPLICATION_JSON.value;
            }else{
                httpRequestOption.contentType = Request.ContentType.APPLICATION_X_WWW_FORM_URLENCODED.value;
            }
            Request.ContentType contentType = Request.ContentType.getInstanceByValue(httpRequestOption.contentType);
            if(null==contentType){
                return;
            }
            flowContext.putTemporaryInstanceData(contentType);
            switch (contentType){
                case MULTIPART_FORMDATA:{
                    if (null == httpRequestOption.boundary) {
                        httpRequestOption.boundary = mimeBoundary();
                    }
                    httpRequestContext.contentType = "multipart/form-data; boundary=" + httpRequestOption.boundary;
                    Set<Map.Entry<String, List<DataFileContext>>> entrySet = httpRequestOption.dataFileMap.entrySet();
                    for (Map.Entry<String, List<DataFileContext>> entry : entrySet) {
                        for (DataFileContext dataFileContext : entry.getValue()) {
                            if (null == dataFileContext.mimeType) {
                                dataFileContext.mimeType = Files.probeContentType(dataFileContext.filePath);
                            }
                        }
                    }
                }break;
                case APPLICATION_JSON:{
                    httpRequestContext.contentType = (httpRequestOption.userContentType==null?"application/json":httpRequestOption.userContentType.value) + "; charset=" + httpRequestOption.charset;
                }break;
                case APPLICATION_X_WWW_FORM_URLENCODED:{
                    httpRequestContext.contentType = "application/x-www-form-urlencoded; charset=" + httpRequestOption.charset;
                    if (!httpRequestOption.dataMap.isEmpty()) {
                        StringBuilder formBuilder = new StringBuilder();
                        Set<Map.Entry<String, Object>> entrySet = httpRequestOption.dataMap.entrySet();
                        for (Map.Entry<String, Object> entry : entrySet) {
                            Object value = entry.getValue();
                            if (null != value) {
                                value = URLEncoder.encode(value.toString(), httpRequestOption.charset);
                            }
                            formBuilder.append(URLEncoder.encode(entry.getKey(), httpRequestOption.charset) + "=" + value + "&");
                        }
                        formBuilder.deleteCharAt(formBuilder.length() - 1);
                        httpRequestOption.requestBody = formBuilder.toString().getBytes(Charset.forName(httpRequestOption.charset));
                    }
                }break;
            }
        }else{
            httpRequestContext.contentType = httpRequestOption.contentType;
        }
    }

    /**
     * 创建随机Boundary字符串作为分隔符
     */
    private static String mimeBoundary() {
        final StringBuilder mime = new StringBuilder(boundaryLength);
        for (int i = 0; i < boundaryLength; i++) {
            mime.append(mimeBoundaryChars[rand.nextInt(mimeBoundaryChars.length)]);
        }
        return mime.toString();
    }
}
