package cn.schoolwow.quickhttp.module.request.execute.domain;

import java.io.IOException;
import java.io.InputStream;

public class ChunkedInputStream extends InputStream {
    private final InputStream inputStream;
    //是否读取到chunk结束
    private boolean endOfChunks;
    //chunk块大小
    private int chunkSize;
    //剩余未读取的chunk块大小
    private int chunkBytesLeft;

    public ChunkedInputStream(InputStream inputStream) {
        this.inputStream = inputStream;
        this.endOfChunks = false;
        this.chunkSize = 0;
        this.chunkBytesLeft = 0;
    }

    @Override
    public void mark(int readAheadLimit){
        inputStream.mark(readAheadLimit);
    }

    @Override
    public void reset() throws IOException {
        inputStream.reset();
    }

    @Override
    public int read() throws IOException {
        if (endOfChunks) {
            return -1;
        }

        if (chunkBytesLeft == 0) {
            parseNextChunk();
            if (endOfChunks) {
                return -1;
            }
        }

        int data = inputStream.read();
        if (data != -1) {
            chunkBytesLeft--;
        }

        if (chunkBytesLeft == 0) {
            // Read CRLF after the chunk
            int crlf1 = inputStream.read();
            int crlf2 = inputStream.read();
            if (crlf1 != '\r' || crlf2 != '\n') {
                throw new IOException("chunk编码格式错误!chunke数据读取完毕后无换行符信息!");
            }
        }

        return data;
    }

    private void parseNextChunk() throws IOException {
        StringBuilder chunkSizeStr = new StringBuilder();
        int data;
        while ((data = inputStream.read()) != -1) {
            char c = (char) data;
            if (c == '\r') {
                // Read '\n' after '\r'
                inputStream.read();
                break;
            }
            chunkSizeStr.append(c);
        }

        if (chunkSizeStr.length() == 0) {
            endOfChunks = true;
            return;
        }

        try {
            chunkSize = Integer.parseInt(chunkSizeStr.toString(), 16);
        } catch (NumberFormatException e) {
            throw new IOException("chunk块大小读取错误!读取字符串:"+chunkSizeStr.toString());
        }

        if (chunkSize == 0) {
            // Last chunk
            endOfChunks = true;
            return;
        }

        chunkBytesLeft = chunkSize;
    }
}
