package cn.schoolwow.quickhttp.module.client;

import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickhttp.domain.client.QuickHttpClientConfig;
import cn.schoolwow.quickhttp.domain.execute.HttpExecuteEngine;
import cn.schoolwow.quickhttp.domain.option.ExecuteFlowListener;
import cn.schoolwow.quickhttp.domain.option.QuickHttpClientListener;
import cn.schoolwow.quickhttp.module.client.okhttp.OkHttpExecuteInterceptor;
import cn.schoolwow.quickhttp.module.common.HttpClientOption;
import okhttp3.OkHttpClient;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSocketFactory;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.ThreadPoolExecutor;

public class QuickHttpClientConfigImpl implements QuickHttpClientConfig {
    private QuickFlow quickFlow;
    private HttpClientOption httpClientOption;

    public QuickHttpClientConfigImpl(QuickFlow quickFlow) {
        this.quickFlow = quickFlow;
        this.httpClientOption = quickFlow.getContextInstanceData(HttpClientOption.class);
    }

    @Override
    public QuickHttpClientConfig debug(boolean debug) {
        httpClientOption.debug = debug;
        return this;
    }

    @Override
    public QuickHttpClientConfig proxy(Proxy proxy) {
        httpClientOption.proxy = proxy;
        return this;
    }

    @Override
    public QuickHttpClientConfig proxy(String host, int port) {
        httpClientOption.proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(host, port));
        return this;
    }

    @Override
    public QuickHttpClientConfig proxyAuthority(String username, String password) {
        httpClientOption.proxyUsername = username;
        httpClientOption.proxyPassword = password;
        return this;
    }

    @Override
    public QuickHttpClientConfig connectTimeout(int connectTimeoutMillis) {
        httpClientOption.connectTimeoutMillis = connectTimeoutMillis;
        return this;
    }

    @Override
    public QuickHttpClientConfig readTimeout(int readTimeoutMillis) {
        httpClientOption.readTimeoutMillis = readTimeoutMillis;
        return this;
    }

    @Override
    public QuickHttpClientConfig followRedirects(boolean followRedirects) {
        httpClientOption.followRedirects = followRedirects;
        return this;
    }

    @Override
    public QuickHttpClientConfig maxFollowRedirectTimes(int maxFollowRedirectTimes) {
        httpClientOption.maxFollowRedirectTimes = maxFollowRedirectTimes;
        return this;
    }

    @Override
    public QuickHttpClientConfig ignoreHttpErrors(boolean ignoreHttpErrors) {
        httpClientOption.ignoreHttpErrors = ignoreHttpErrors;
        return this;
    }

    @Override
    public QuickHttpClientConfig retryTimes(int retryTimes) {
        httpClientOption.retryTimes = retryTimes;
        return this;
    }

    @Override
    public QuickHttpClientConfig hostnameVerifier(HostnameVerifier hostnameVerifier) {
        httpClientOption.hostnameVerifier = hostnameVerifier;
        return this;
    }

    @Override
    public QuickHttpClientConfig sslSocketFactory(SSLSocketFactory sslSocketFactory) {
        httpClientOption.sslSocketFactory = sslSocketFactory;
        return this;
    }

    @Override
    public QuickHttpClientConfig threadPoolExecutor(ThreadPoolExecutor threadPoolExecutor) {
        httpClientOption.threadPoolExecutor = threadPoolExecutor;
        return this;
    }

    @Override
    public QuickHttpClientConfig origin(String origin) {
        httpClientOption.origin = origin;
        return this;
    }

    @Override
    public QuickHttpClientConfig addHeader(String name, String value) {
        if(!httpClientOption.globalHeaderMap.containsKey(name)){
            httpClientOption.globalHeaderMap.put(name, new ArrayList<>());
        }
        httpClientOption.globalHeaderMap.get(name).add(value);
        return this;
    }

    @Override
    public QuickHttpClientConfig setHeader(String name, String value) {
        httpClientOption.globalHeaderMap.put(name, Arrays.asList(value));
        return this;
    }

    @Override
    public QuickHttpClientConfig quickHttpClientListener(QuickHttpClientListener quickHttpClientListener) {
        httpClientOption.quickHttpClientListener = quickHttpClientListener;
        return this;
    }

    @Override
    public QuickHttpClientConfig executeFlowListener(ExecuteFlowListener executeFlowListener) {
        httpClientOption.executeFlowListener = executeFlowListener;
        return this;
    }

    @Override
    public QuickHttpClientConfig httpExecuteEngine(HttpExecuteEngine httpExecuteEngine) {
        httpClientOption.httpExecuteEngine = httpExecuteEngine;
        switch (httpExecuteEngine){
            case OkHttp:{
                OkHttpClient okHttpClient = new OkHttpClient.Builder()
                        .addInterceptor(new OkHttpExecuteInterceptor(quickFlow))
                        .build();
                quickFlow.putContextTemporaryInstanceData(okHttpClient, OkHttpClient.class);

                if(null!=httpClientOption.executeFlowListener){
                    httpClientOption.executeFlowListener.afterCreateExecuteFlow(quickFlow);
                }
            }break;
        }
        return this;
    }

}
