package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowConfig;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class QuickFlowExecutorInner {
    private FlowExecutorConfig flowExecutorConfig;

    private FlowExecutorRootConfig flowExecutorRootConfig;

    private FlowConfig flowConfig;

    public QuickFlowExecutorInner(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
        this.flowExecutorRootConfig = flowExecutorConfig.flowExecutorRootConfig;
        this.flowConfig = this.flowExecutorRootConfig.flowConfig;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext doExecute() {
        setRequestDataMap();
        flowExecutorConfig.flowNameList.addAll(QuickFlowUtil.getFlowNameList(flowExecutorConfig));
        try {
            beforeExecuteBusiness();
            executeBusiness();
        } catch (Exception e) {
            handleBusinessException(e);
        } finally {
            handleBusinessFinally();
        }
        return flowExecutorConfig.flowContext;
    }

    /**固定请求参数*/
    private void setRequestDataMap(){
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String requestKey:flowExecutorRootConfig.requestKeySet){
            requestDataMap.put(requestKey, flowExecutorRootConfig.dataMap.get(requestKey));
        }
        flowExecutorRootConfig.requestKeyMap = new JSONObject(requestDataMap);
    }

    /**执行流程列表之前*/
    private void beforeExecuteBusiness(){
        //添加全局参数
        flowExecutorRootConfig.dataMap.putAll(flowConfig.dataMap);
        if(!flowExecutorConfig.compositeBusiness&&null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.recordContent("当前流程是主流程!流程id:"+flowExecutorRootConfig.id);
            flowExecutorConfig.printFlowTrace("流程id:"+flowExecutorRootConfig.id);
        }
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "try", null);
        executeBeforeAfterFlowHandler(true, flowExecutorConfig.beforeAfterFlowHandler);
        {
            boolean shouldExecuteGlobalTryCatchFinally = getShouldExecuteGlobalTryCatchFinally();
            flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-try");
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "try", null);
        }
        {
            boolean shouldExecuteGlobalBeforeAfter = getShouldExecuteGlobalBeforeAfter();
            flowExecutorConfig.recordContent((shouldExecuteGlobalBeforeAfter?"执行":"不执行")+"全局beforeAfter处理器");
            executeBeforeAfterFlowHandler(shouldExecuteGlobalBeforeAfter, flowConfig.beforeAfterFlowHandler);
        }
        flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        flowExecutorConfig.flowExecutorRootConfig.incrementFlowLevel();
    }

    private void executeBusiness() throws Exception {
        for (BusinessFlow flow : flowExecutorConfig.flowList) {
            flowExecutorConfig.getRemarkBuilder().setLength(0);
            flowExecutorConfig.currentFlow = flow;

            long startTime = System.currentTimeMillis();
            try {
                executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "before");
                {
                    boolean shouldExecuteGlobalSingleFlow = getShouldExecuteGlobalSingleFlow();
                    flowExecutorConfig.recordContent((shouldExecuteGlobalSingleFlow?"执行":"不执行")+"全局singleFlow-before处理器");
                    executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "before");
                }
                startTime = System.currentTimeMillis();
                flow.executeBusinessFlow(flowExecutorConfig.flowContext);
                long endTime = System.currentTimeMillis();
                if(!flow.name().startsWith("composite:")&&flowExecutorConfig.flowList.size()>1){
                    flowExecutorConfig.recordContent("当前流程不是复合流程或者流程列表有且仅有一个复合流程");
                    flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|" + flow.name() + flowExecutorConfig.getRemarkBuilder().toString());
                }else{
                    flowExecutorConfig.recordContent("当前流程是复合流程");
                }
            }catch (BrokenCurrentFlowException e){
                long endTime = System.currentTimeMillis();
                flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
            }
            executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "after");
            {
                boolean shouldExecuteGlobalSingleFlow = getShouldExecuteGlobalSingleFlow();
                flowExecutorConfig.recordContent((shouldExecuteGlobalSingleFlow?"执行":"不执行")+"全局singleFlow-after处理器");
                executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "after");
            }
        }
    }

    private void handleBusinessException(Exception e){
        Exception targetException = e;
        QuickFlowRuntimeException quickFlowRuntimeException = null;
        if(e instanceof QuickFlowRuntimeException){
            targetException = ((QuickFlowRuntimeException)e).targetException;
            quickFlowRuntimeException = (QuickFlowRuntimeException) e;
        }else{
            quickFlowRuntimeException = new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
        }
        //判断异常类型
        if(targetException instanceof BrokenException){
            handleBrokenException((BrokenException) targetException);
        }else{
            handleException(e, quickFlowRuntimeException);
        }
    }

    private void handleBrokenException(BrokenException e){
        boolean shouldIgnoreBroken = getShouldIgnoreBroken();
        flowExecutorConfig.recordContent((shouldIgnoreBroken?"忽略":"不忽略")+"中断异常");
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程发生中断|原因:"+e.getReason()+(shouldIgnoreBroken?"|忽略该中断":"")+ flowExecutorConfig.getRemarkBuilder().toString());
        }else{
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.brokenReason = e.getReason();
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程终止|原因:"+e.getReason()+flowExecutorConfig.getRemarkBuilder().toString());
        }
        if(!shouldIgnoreBroken){
            throw e;
        }
    }

    private void handleException(Exception targetException, QuickFlowRuntimeException quickFlowRuntimeException){
        boolean shouldIgnoreException = getShouldIgnoreException();
        flowExecutorConfig.recordContent((shouldIgnoreException?"忽略":"不忽略")+"抛出异常");
        String exceptionMessage = targetException.getClass().getName() + "-" + targetException.getMessage();
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            //非主流程
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
        }else{
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.exception = targetException;
            {
                boolean shouldExecuteGlobalTryCatchFinally = getShouldExecuteGlobalTryCatchFinally();
                flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-catch");
                executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "catch", targetException);
            }
        }
        if(!shouldIgnoreException){
            throw quickFlowRuntimeException;
        }
    }

    private void handleBusinessFinally(){
        flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "finally", null);
        {
            boolean shouldExecuteGlobalTryCatchFinally = getShouldExecuteGlobalTryCatchFinally();
            flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-finally");
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "finally", null);
        }
        if(flowExecutorConfig.flowList.size()>1){
            flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        }
        flowExecutorConfig.recordContent("================");
    }

    private void executeTryCatchFinallyHandler(boolean shouldExecute, TryCatchFinallyHandler tryCatchFinallyHandler, String operation, Exception e) {
        if(null==tryCatchFinallyHandler||!shouldExecute){
            return;
        }
        long startTime = System.currentTimeMillis();
        try {
            switch (operation){
                case "try":{tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);}break;
                case "catch":{tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);}break;
                case "finally":{
                    if(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler==tryCatchFinallyHandler){
                        flowExecutorConfig.printFlowTraceWithIndex(operation+":"+tryCatchFinallyHandler.name());
                    }
                    tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
                }break;
                default:{
                    throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
                }
            }
            long endTime = System.currentTimeMillis();
            flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|"+tryCatchFinallyHandler.name()+":"+operation);
        }catch (Exception ex){
            if(ex instanceof QuickFlowRuntimeException){
                throw (QuickFlowRuntimeException)ex;
            }else{
                throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, ex);
            }
        }
    }

    private void executeBeforeAfterFlowHandler(boolean shouldExecute, BeforeAfterFlowHandler beforeAfterFlowHandler) {
        if(null==beforeAfterFlowHandler||!shouldExecute){
            return;
        }
        List<BusinessFlow> beforeFlowList = beforeAfterFlowHandler.getBeforeFlowList();
        if(null!=beforeFlowList&&!beforeFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(0, beforeFlowList);
        }
        List<BusinessFlow> afterFlowList = beforeAfterFlowHandler.getAfterFlowList();
        if(null!=afterFlowList&&!afterFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(afterFlowList);
        }
    }

    private void executeSingleFlowHandler(boolean shouldExecute, BusinessFlow flow, SingleFlowListener singleFlowListener, String operation) throws Exception {
        if(null==singleFlowListener||!shouldExecute){
            return;
        }
        switch (operation){
            case "before":{
                singleFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowContext);
            };break;
            case "after":{
                singleFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowContext);
            };break;
            default:{
                throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
            }
        }
    }

    private boolean getShouldIgnoreBroken(){
        if(null!=flowExecutorConfig.ignoreBroken){
            return flowExecutorConfig.ignoreBroken;
        }
        return flowConfig.ignoreBroken;
    }

    private boolean getShouldIgnoreException(){
        if(null!=flowExecutorConfig.ignoreException){
            return flowExecutorConfig.ignoreException;
        }
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            return false;
        }
        return flowConfig.ignoreException;
    }

    private boolean getShouldExecuteGlobalTryCatchFinally(){
        if(null!=flowExecutorConfig.executeGlobalTryCatchFinally){
            return flowExecutorConfig.executeGlobalTryCatchFinally;
        }
        if(flowExecutorConfig.compositeBusiness){
            return false;
        }
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            return true;
        }
        return flowConfig.executeGlobalTryCatchFinally;
    }

    private boolean getShouldExecuteGlobalBeforeAfter(){
        if(null!=flowExecutorConfig.executeGlobalBeforeAfter){
            return flowExecutorConfig.executeGlobalBeforeAfter;
        }
        if(flowExecutorConfig.compositeBusiness){
            return false;
        }
        if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowConfig.beforeAfterFlowHandler){
            return true;
        }
        return flowConfig.executeGlobalBeforeAfter;
    }

    private boolean getShouldExecuteGlobalSingleFlow(){
        if(null!=flowExecutorConfig.executeGlobalSingleFlow){
            return flowExecutorConfig.executeGlobalSingleFlow;
        }
        if(flowExecutorConfig.compositeBusiness){
            return false;
        }
        if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowConfig.singleFlowListener){
            return true;
        }
        return flowConfig.executeGlobalSingleFlow;
    }
}
