package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorOption;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

public class QuickFlowExecutor {
    private Logger logger = LoggerFactory.getLogger(QuickFlowExecutor.class);

    /**配置信息*/
    private FlowExecutorConfig flowExecutorConfig;

    public QuickFlowExecutor(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    /**
     * 设置是否开启日志记录
     */
    public QuickFlowExecutor startRecord(boolean startRecord) {
        flowExecutorConfig.flowExecutorRootConfig.startRecord = startRecord;
        return this;
    }

    /**
     * 设置请求参数
     */
    public QuickFlowExecutor putData(String key, Object value) {
        flowExecutorConfig.flowContext.putData(key, value);
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.flowExecutorRootConfig.requestKeySet.add(key);
        }
        return this;
    }

    /**
     * 设置请求参数
     */
    public QuickFlowExecutor putData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putData(dataMap);
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        }
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putContextData(String key, Object value) {
        flowExecutorConfig.flowContext.putData(key, value);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putContextData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putData(dataMap);
        return this;
    }

    /**
     * 设置临时请求参数
     */
    public QuickFlowExecutor putTemporaryData(String key, Object value) {
        flowExecutorConfig.flowContext.putTemporaryData(key, value);
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.flowExecutorRootConfig.requestKeySet.add(key);
        }
        return this;
    }

    /**
     * 设置临时请求参数
     */
    public QuickFlowExecutor putTemporaryData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putTemporaryData(dataMap);
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        }
        return this;
    }

    /**
     * 设置线程本地参数
     */
    public QuickFlowExecutor putThreadLocalData(String key, Object value) {
        flowExecutorConfig.flowContext.putThreadLocalData(key, value);
        return this;
    }

    /**
     * 设置线程本地参数
     */
    public QuickFlowExecutor putThreadLocalData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putThreadLocalData(dataMap);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(String flowName) {
        if(null==flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        BusinessFlow flow = flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowMappingHandler.getFlow(flowName);
        if(null==flow){
            CompositeBusinessFlow compositeBusinessFlow = flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowMappingHandler.getCompositeBusinessFlow(flowName);
            if(null==compositeBusinessFlow){
                throw new IllegalArgumentException("根据流程名称查找流程示例失败!流程名称:"+flowName);
            }
            next(compositeBusinessFlow);
        }else{
            flowExecutorConfig.flowList.add(flow);
        }
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(BusinessFlow flow) {
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(CompositeBusinessFlow compositeBusinessFlow) {
        BusinessFlow businessFlow = QuickFlowUtil.getCompositeBusinessFlow(compositeBusinessFlow);
        flowExecutorConfig.flowList.add(businessFlow);
        return this;
    }

    /**
     * 是否打印路径
     */
    public QuickFlowExecutor printTrace(boolean printTrace) {
        flowExecutorConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略中断
     */
    public QuickFlowExecutor ignoreBroken(boolean ignoreBroken) {
        flowExecutorConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    /**
     * 是否忽略异常
     */
    public QuickFlowExecutor ignoreException(boolean ignoreException) {
        flowExecutorConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 是否所有流程都执行全局tryCatchFinally处理器
     */
    public QuickFlowExecutor executeGlobalTryCatchFinally(boolean executeGlobalTryCatchFinally) {
        flowExecutorConfig.executeGlobalTryCatchFinally = executeGlobalTryCatchFinally;
        return this;
    }

    /**
     * 是否所有流程都执行全局beforeAfter处理器
     */
    public QuickFlowExecutor executeGlobalBeforeAfter(boolean executeGlobalBeforeAfter) {
        flowExecutorConfig.executeGlobalBeforeAfter = executeGlobalBeforeAfter;
        return this;
    }

    /**
     * 是否所有流程都执行全局singleFlow处理器
     */
    public QuickFlowExecutor executeGlobalSingleFlow(boolean executeGlobalSingleFlow) {
        flowExecutorConfig.executeGlobalSingleFlow = executeGlobalSingleFlow;
        return this;
    }

    /**
     * 流程执行前后事件监听
     */
    public QuickFlowExecutor singleFlowListener(SingleFlowListener singleFlowListener) {
        flowExecutorConfig.singleFlowListener = singleFlowListener;
        return this;
    }

    /**
     * 主流程列表前后添加流程列表
     */
    public QuickFlowExecutor beforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        flowExecutorConfig.beforeAfterFlowHandler = beforeAfterFlowHandler;
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlowExecutor tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowExecutorConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 获取流程执行器信息
     * */
    public FlowExecutorOption getOption(){
        FlowExecutorOption flowExecutorOption = new FlowExecutorOption();
        flowExecutorOption.name = flowExecutorConfig.name;
        flowExecutorOption.requestData = new JSONObject(flowExecutorConfig.flowExecutorRootConfig.dataMap);
        flowExecutorOption.flowNameList = QuickFlowUtil.getFlowNameList(flowExecutorConfig);
        return flowExecutorOption;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext execute() {
        QuickFlowExecutorInner quickFlowExecutorInner = new QuickFlowExecutorInner(flowExecutorConfig);
        return quickFlowExecutorInner.doExecute();
    }

    public FlowExecutorConfig getFlowExecutorConfig() {
        return flowExecutorConfig;
    }

    public FlowExecutorRootConfig getFlowExecutorRootConfig() {
        return flowExecutorConfig.flowExecutorRootConfig;
    }
}
