package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowConfig;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentCompositeBusinessException;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class QuickFlowExecutorInner {
    private FlowExecutorConfig flowExecutorConfig;

    private FlowExecutorRootConfig flowExecutorRootConfig;

    private FlowConfig flowConfig;

    public QuickFlowExecutorInner(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
        this.flowExecutorRootConfig = flowExecutorConfig.flowExecutorRootConfig;
        this.flowConfig = this.flowExecutorRootConfig.flowConfig;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext doExecute() {
        setRequestDataMap();
        flowExecutorConfig.flowNameList.addAll(QuickFlowUtil.getFlowNameList(flowExecutorConfig));
        try {
            beforeExecuteBusiness();
            executeBusiness();
        } catch (Exception e) {
            handleBusinessException(e);
        } finally {
            handleBusinessFinally();
        }
        return flowExecutorConfig.flowContext;
    }

    /**固定请求参数*/
    private void setRequestDataMap(){
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String requestKey:flowExecutorRootConfig.requestKeySet){
            requestDataMap.put(requestKey, flowExecutorRootConfig.dataMap.get(requestKey));
        }
        flowExecutorRootConfig.requestKeyMap = new JSONObject(requestDataMap);
    }

    /**执行流程列表之前*/
    private void beforeExecuteBusiness(){
        //添加全局参数,若存在则跳过
        for(String key:flowConfig.dataMap.keySet()){
            if(!flowExecutorRootConfig.dataMap.containsKey(key)){
                flowExecutorRootConfig.dataMap.put(key, flowConfig.dataMap.get(key));
            }
        }
        if(!flowExecutorConfig.compositeBusiness&&null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.recordContent("当前流程是主流程!流程id:"+flowExecutorRootConfig.id);
            flowExecutorConfig.printFlowTrace("流程id:"+flowExecutorRootConfig.id);
        }
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "try", null);
        executeBeforeAfterFlowHandler(true, flowExecutorConfig.beforeAfterFlowHandler);
        {
            boolean shouldExecuteGlobalTryCatchFinally = getGlobalHandlerShouldExecute("TryCatchFinally");
            flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-try");
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "try", null);
        }
        {
            boolean shouldExecuteGlobalBeforeAfter = getGlobalHandlerShouldExecute("BeforeAfter");
            flowExecutorConfig.recordContent((shouldExecuteGlobalBeforeAfter?"执行":"不执行")+"全局beforeAfter处理器");
            executeBeforeAfterFlowHandler(shouldExecuteGlobalBeforeAfter, flowConfig.beforeAfterFlowHandler);
        }
        flowExecutorConfig.flowExecutorRootConfig.incrementFlowLevel();
        //子流程列表大于1或者只有一个流程但是该流程是复合流程，则打印当前流程名称
        if(flowExecutorConfig.flowList.size()>1||(flowExecutorConfig.flowList.size()==1&&flowExecutorConfig.flowList.get(0).name().startsWith("composite:"))){
            flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        }
    }

    private void executeBusiness() throws Exception {
        for (BusinessFlow flow : flowExecutorConfig.flowList) {
            flowExecutorConfig.getRemarkBuilder().setLength(0);
            flowExecutorConfig.currentFlow = flow;
            QuickFlowUtil.loadFlowConfig(flow, flowExecutorConfig);

            long startTime = System.currentTimeMillis();
            try {
                executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "before");
                {
                    boolean shouldExecuteGlobalSingleFlow = getGlobalHandlerShouldExecute("SingleFlow");
                    flowExecutorConfig.recordContent((shouldExecuteGlobalSingleFlow?"执行":"不执行")+"全局singleFlow-before处理器");
                    executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "before");
                }
                startTime = System.currentTimeMillis();
                flow.executeBusinessFlow(flowExecutorConfig.flowContext);
                long endTime = System.currentTimeMillis();
                if(!flow.name().startsWith("composite:")){
                    flowExecutorConfig.recordContent("当前流程不是复合流程");
                    flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|" + flow.name() + flowExecutorConfig.getRemarkBuilder().toString());
                }else{
                    flowExecutorConfig.recordContent("当前流程是复合流程");
                }
            }catch (BrokenCurrentFlowException e){
                long endTime = System.currentTimeMillis();
                flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
            }catch (BrokenCurrentCompositeBusinessException e){
                long endTime = System.currentTimeMillis();
                flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|[x]|"+ flow.name() + "|当前复合流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
                flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
                break;
            }
            executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "after");
            {
                boolean shouldExecuteGlobalSingleFlow = getGlobalHandlerShouldExecute("SingleFlow");
                flowExecutorConfig.recordContent((shouldExecuteGlobalSingleFlow?"执行":"不执行")+"全局singleFlow-after处理器");
                executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "after");
            }
            flowExecutorConfig.currentFlowDataMap.clear();
        }
        flowExecutorConfig.currentCompositeFlowDataMap.get().clear();
    }



    private void handleBusinessException(Exception e){
        flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
        Exception targetException = e;
        QuickFlowRuntimeException quickFlowRuntimeException = null;
        if(e instanceof QuickFlowRuntimeException){
            targetException = ((QuickFlowRuntimeException)e).targetException;
            quickFlowRuntimeException = (QuickFlowRuntimeException) e;
        }else{
            quickFlowRuntimeException = new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
        }
        //判断异常类型
        if(targetException instanceof BrokenException){
            handleBrokenException((BrokenException) targetException);
        }else{
            handleException(e, quickFlowRuntimeException);
        }
    }

    private void handleBrokenException(BrokenException e){
        boolean shouldIgnoreBroken = getShouldIgnoreBroken();
        flowExecutorConfig.recordContent((shouldIgnoreBroken?"忽略":"不忽略")+"中断异常");
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程发生中断|原因:"+e.getReason()+(shouldIgnoreBroken?"|忽略该中断":"")+ flowExecutorConfig.getRemarkBuilder().toString());
        }else{
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.brokenReason = e.getReason();
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程终止|原因:"+e.getReason()+flowExecutorConfig.getRemarkBuilder().toString());
        }
        if(!shouldIgnoreBroken){
            throw e;
        }
    }

    private void handleException(Exception targetException, QuickFlowRuntimeException quickFlowRuntimeException){
        boolean shouldIgnoreException = getShouldIgnoreException();
        flowExecutorConfig.recordContent((shouldIgnoreException?"忽略":"不忽略")+"抛出异常");
        String exceptionMessage = targetException.getClass().getName() + "-" + targetException.getMessage();
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            //非主流程
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
        }else{
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.exception = targetException;
            {
                boolean shouldExecuteGlobalTryCatchFinally = getGlobalHandlerShouldExecute("TryCatchFinally");
                flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-catch");
                executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "catch", targetException);
            }
        }
        if(!shouldIgnoreException){
            throw quickFlowRuntimeException;
        }
    }

    private void handleBusinessFinally(){
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "finally", null);
        {
            boolean shouldExecuteGlobalTryCatchFinally = getGlobalHandlerShouldExecute("TryCatchFinally");
            flowExecutorConfig.recordContent((shouldExecuteGlobalTryCatchFinally?"执行":"不执行")+"全局tryCatchFinally处理器-finally");
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "finally", null);
        }
        //子流程列表大于1或者只有一个流程但是该流程是复合流程，则打印当前流程名称
        if(flowExecutorConfig.flowList.size()>1||(flowExecutorConfig.flowList.size()==1&&flowExecutorConfig.flowList.get(0).name().startsWith("composite:"))){
            flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        }
        flowExecutorConfig.recordContent("================");
        if(null==flowExecutorConfig.flowExecutorRootConfig.exception&&null==flowExecutorConfig.flowExecutorRootConfig.brokenReason){
            flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
        }
    }

    private void executeTryCatchFinallyHandler(boolean shouldExecute, TryCatchFinallyHandler tryCatchFinallyHandler, String operation, Exception e) {
        if(null==tryCatchFinallyHandler||!shouldExecute){
            return;
        }
        long startTime = System.currentTimeMillis();
        try {
            switch (operation){
                case "try":{tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);}break;
                case "catch":{tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);}break;
                case "finally":{
                    if(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler==tryCatchFinallyHandler){
                        flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
                        flowExecutorConfig.printFlowTraceWithIndex(operation+":" + tryCatchFinallyHandler.name());
                    }
                    tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
                }break;
                default:{
                    throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
                }
            }
            long endTime = System.currentTimeMillis();
            flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|"+operation + ":" + tryCatchFinallyHandler.name());
        }catch (Exception ex){
            if(ex instanceof QuickFlowRuntimeException){
                throw (QuickFlowRuntimeException)ex;
            }else{
                throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, ex);
            }
        }
    }

    private void executeBeforeAfterFlowHandler(boolean shouldExecute, BeforeAfterFlowHandler beforeAfterFlowHandler) {
        if(null==beforeAfterFlowHandler||!shouldExecute){
            return;
        }
        List<BusinessFlow> beforeFlowList = beforeAfterFlowHandler.getBeforeFlowList();
        if(null!=beforeFlowList&&!beforeFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(0, beforeFlowList);
        }
        List<BusinessFlow> afterFlowList = beforeAfterFlowHandler.getAfterFlowList();
        if(null!=afterFlowList&&!afterFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(afterFlowList);
        }
    }

    private void executeSingleFlowHandler(boolean shouldExecute, BusinessFlow flow, SingleFlowListener singleFlowListener, String operation) throws Exception {
        if(null==singleFlowListener||!shouldExecute){
            return;
        }
        switch (operation){
            case "before":{
                singleFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowContext);
            };break;
            case "after":{
                singleFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowContext);
            };break;
            default:{
                throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
            }
        }
    }

    private boolean getShouldIgnoreBroken(){
        if(null!=flowExecutorConfig.ignoreBroken){
            return flowExecutorConfig.ignoreBroken;
        }
        return flowConfig.ignoreBroken;
    }

    private boolean getShouldIgnoreException(){
        if(null!=flowExecutorConfig.ignoreException){
            return flowExecutorConfig.ignoreException;
        }
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            return false;
        }
        return flowConfig.ignoreException;
    }

    private boolean getGlobalHandlerShouldExecute(String type){
        //首选当前流程设置标志
        switch (type){
            case "TryCatchFinally":{
                if(null!=flowExecutorConfig.executeGlobalTryCatchFinally){
                    return flowExecutorConfig.executeGlobalTryCatchFinally;
                }
            }break;
            case "BeforeAfter":{
                if(null!=flowExecutorConfig.executeGlobalBeforeAfter){
                    return flowExecutorConfig.executeGlobalBeforeAfter;
                }
            }break;
            case "SingleFlow":{
                if(null!=flowExecutorConfig.executeGlobalSingleFlow){
                    return flowExecutorConfig.executeGlobalSingleFlow;
                }
            }break;
            default:{
                throw new IllegalArgumentException("框架内部错误!不支持的事件类型!类型名:"+type);
            }
        }
        //其次最外层流程一定执行
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            return true;
        }
        //然后若用户设置全局执行则一定执行
        switch (type){
            case "TryCatchFinally":{
                if(flowConfig.executeGlobalTryCatchFinally){
                    return true;
                }
            }break;
            case "BeforeAfter":{
                if(flowConfig.executeGlobalBeforeAfter){
                    return true;
                }
            }break;
            case "SingleFlow":{
                if(flowConfig.executeGlobalSingleFlow){
                    return true;
                }
            }break;
            default:{
                throw new IllegalArgumentException("框架内部错误!不支持的事件类型!类型名:"+type);
            }
        }
        //复合流程不执行
        if(flowExecutorConfig.compositeBusiness){
            return false;
        }
        //其他情况不执行
        return false;
    }

}
