package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.*;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.listener.*;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.Map;

public class QuickFlow {
    /**流程配置信息*/
    private FlowConfig flowConfig = new FlowConfig();

    /**默认流程配置对象*/
    private static QuickFlow defaultQuickFlow = new QuickFlow();

    public static QuickFlow newInstance() {
        return new QuickFlow();
    }

    /**获取默认QuickFlow对象*/
    public static QuickFlow defaultQuickFlow(){
        return defaultQuickFlow;
    }

    /**执行静态业务逻辑(仅执行一次)*/
    public static QuickFlow executeStaticFlowList(BusinessFlow... businessFlows){
        defaultQuickFlow.executeFlow(businessFlows);
        return defaultQuickFlow();
    }

    /**
     * 是否开启调试模式
     */
    public QuickFlow debug(boolean debug) {
        flowConfig.debug = debug;
        return this;
    }

    /**
     * 是否打印执行路径,默认为true
     */
    public QuickFlow printTrace(boolean printTrace) {
        flowConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 流程日志记录层级,默认为1
     */
    public QuickFlow traceLevel(int traceLevel) {
        flowConfig.traceLevel = traceLevel;
        return this;
    }

    /**
     * 是否忽略中断,默认为true
     */
    public QuickFlow ignoreBroken(boolean ignoreBroken) {
        flowConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    /**
     * 是否忽略异常,默认为false
     */
    public QuickFlow ignoreException(boolean ignoreException) {
        flowConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 获取全局上下文参数
     */
    public Object getContextData(String key) {
        return flowConfig.dataMap.get(key);
    }

    /**
     * 获取全局上下文参数
     */
    public Map<String,Object> getContextDataMap() {
        return flowConfig.dataMap;
    }

    /**
     * 设置全局上下文参数
     */
    public QuickFlow putContextData(String key, Object value) {
        flowConfig.dataMap.put(key, value);
        return this;
    }

    /**
     * 设置全局上下文参数
     */
    public QuickFlow putContextDataMap(Map<String,Object> dataMap) {
        flowConfig.dataMap.putAll(dataMap);
        return this;
    }

    /**
     * 是否所有流程都执行全局tryCatchFinally处理器
     */
    public QuickFlow executeGlobalTryCatchFinally(boolean executeGlobalTryCatchFinally) {
        flowConfig.executeGlobalTryCatchFinally = executeGlobalTryCatchFinally;
        return this;
    }

    /**
     * 是否所有流程都执行全局beforeAfter处理器
     */
    public QuickFlow executeGlobalBeforeAfter(boolean executeGlobalBeforeAfter) {
        flowConfig.executeGlobalBeforeAfter = executeGlobalBeforeAfter;
        return this;
    }

    /**
     * 是否所有流程都执行全局singleFlow处理器
     */
    public QuickFlow executeGlobalSingleFlow(boolean executeGlobalSingleFlow) {
        flowConfig.executeGlobalSingleFlow = executeGlobalSingleFlow;
        return this;
    }

    /**
     * 指定流程名称和实例映射处理器
     */
    public QuickFlow flowNameInstanceMappingHandler(FlowMappingHandler flowMappingHandler) {
        flowConfig.flowMappingHandler = flowMappingHandler;
        return this;
    }

    /**
     * 流程执行前后事件监听
     */
    public QuickFlow singleFlowListener(SingleFlowListener singleFlowListener) {
        if(null==singleFlowListener){
            return this;
        }
        if(null==singleFlowListener.name()||singleFlowListener.name().isEmpty()){
            throw new IllegalArgumentException("监听器名称不能为空!监听器类名:"+singleFlowListener.getClass().getName());
        }
        flowConfig.singleFlowListener = singleFlowListener;
        return this;
    }

    /**
     * 主流程列表前后添加流程列表
     */
    public QuickFlow beforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        if(null==beforeAfterFlowHandler){
            return this;
        }
        if(null==beforeAfterFlowHandler.name()||beforeAfterFlowHandler.name().isEmpty()){
            throw new IllegalArgumentException("监听器名称不能为空!监听器类名:"+beforeAfterFlowHandler.getClass().getName());
        }
        flowConfig.beforeAfterFlowHandler = beforeAfterFlowHandler;
        return this;
    }

    /**
     * 指定流程配置数据处理器
     */
    public QuickFlow flowConfigRequestDataHandler(FlowConfigRequestDataHandler flowConfigRequestDataHandler) {
        if(null==flowConfigRequestDataHandler){
            return this;
        }
        flowConfig.flowConfigRequestDataHandler = flowConfigRequestDataHandler;
        try {
            Map<String,JSONObject> flowConfigMap = flowConfig.flowConfigRequestDataHandler.loadFlowConfigRequestDataMap();
            if(null!=flowConfigMap&&!flowConfigMap.isEmpty()){
                flowConfig.flowConfigRequestDataMap.putAll(flowConfigMap);
            }
            return this;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    /**
     * 指定try-catch-finally流程列表
     */
    public QuickFlow tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        if(null==tryCatchFinallyHandler){
            return this;
        }
        if(null==tryCatchFinallyHandler.name()||tryCatchFinallyHandler.name().isEmpty()){
            throw new IllegalArgumentException("监听器名称不能为空!监听器类名:"+tryCatchFinallyHandler.getClass().getName());
        }
        flowConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(String name) {
        if(null==name||name.isEmpty()){
            throw new IllegalArgumentException("请指定主流程名称!");
        }
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;
        flowExecutorConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        flowExecutorConfig.currentCompositeFlowDataMap.set(new HashMap<>());

        FlowExecutorRootConfig flowExecutorRootConfig = new FlowExecutorRootConfig();
        flowExecutorRootConfig.threadLocalDataMap.set(new HashMap<>());
        flowExecutorRootConfig.flowConfig = flowConfig;
        flowExecutorRootConfig.quickFlow = this;
        flowExecutorConfig.flowExecutorRootConfig = flowExecutorRootConfig;

        return new QuickFlowExecutor(flowExecutorConfig);
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(BusinessFlow businessFlow) {
        return startFlow(businessFlow.name())
                .next(businessFlow);
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(FlowExecutorOption flowExecutorOption) {
        if(null==flowConfig.flowMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        if(null==flowExecutorOption.name||flowExecutorOption.name.isEmpty()){
            throw new IllegalArgumentException("请先指定流程名称!");
        }
        //首先查找复合流程
        {
            CompositeBusinessFlow compositeBusinessFlow = flowConfig.flowMappingHandler.getCompositeBusinessFlow(flowExecutorOption.name);
            if(null!=compositeBusinessFlow){
                return startFlow(compositeBusinessFlow).putData(flowExecutorOption.requestData);
            }
        }
        //其次查找单个流程
        {
            BusinessFlow businessFlow = flowConfig.flowMappingHandler.getFlow(flowExecutorOption.name);
            if(null!=businessFlow){
                return startFlow(businessFlow).putData(flowExecutorOption.requestData);
            }
        }
        //剩下为自定义流程
        if(null==flowExecutorOption.flowNameList||flowExecutorOption.flowNameList.isEmpty()){
            throw new IllegalArgumentException("请指定流程列表!");
        }
        QuickFlowExecutor quickFlowExecutor = startFlow(flowExecutorOption.name).putData(flowExecutorOption.requestData);
        for(String flowName:flowExecutorOption.flowNameList){
            if(flowName.startsWith("handler-")){
                int indexOf = flowName.indexOf(":")+1;
                String handlerName = flowName.substring(indexOf);
                if(flowName.startsWith("handler-tryCatchFinally:")){
                    TryCatchFinallyHandler tryCatchFinallyHandler = flowConfig.flowMappingHandler.getTryCatchFinallyHandler(handlerName);
                    if(null==tryCatchFinallyHandler){
                        throw new IllegalArgumentException("不存在tryCatchFinally处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.tryCatchFinallyHandler(tryCatchFinallyHandler);
                }
                if(flowName.startsWith("handler-beforeAfter:")){
                    BeforeAfterFlowHandler beforeAfterFlowHandler = flowConfig.flowMappingHandler.getBeforeAfterHandler(flowName.substring(indexOf));
                    if(null==beforeAfterFlowHandler){
                        throw new IllegalArgumentException("不存在beforeAfter处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.beforeAfterFlowHandler(beforeAfterFlowHandler);
                }
                if(flowName.startsWith("handler-singleFlow:")){
                    SingleFlowListener singleFlowListener = flowConfig.flowMappingHandler.getSingleFlowListener(flowName.substring(indexOf));
                    if(null==singleFlowListener){
                        throw new IllegalArgumentException("不存在singleFlow处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.singleFlowListener(singleFlowListener);
                }
            } else if(flowName.startsWith("composite:")){
                String compositeFlowName = flowName.substring("composite:".length());
                CompositeBusinessFlow selectedCompositeBusinessFlow = flowConfig.flowMappingHandler.getCompositeBusinessFlow(compositeFlowName);
                if(null==selectedCompositeBusinessFlow){
                    throw new IllegalArgumentException("不存在复合流程!复合流程名称:"+compositeFlowName);
                }
                quickFlowExecutor.next(selectedCompositeBusinessFlow);
            }else{
                BusinessFlow businessFlow = flowConfig.flowMappingHandler.getFlow(flowName);
                if(null==businessFlow){
                    throw new IllegalArgumentException("不存在流程!流程名称:"+flowName);
                }
                quickFlowExecutor.next(businessFlow);
            }
        }
        return quickFlowExecutor;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(CompositeBusinessFlow compositeBusinessFlow) {
        QuickFlowExecutor quickFlowExecutor = startFlow(compositeBusinessFlow.name());
        compositeBusinessFlow.executeCompositeBusiness(quickFlowExecutor);
        QuickFlowUtil.loadFlowConfigByCompositeBusiness(compositeBusinessFlow, quickFlowExecutor.getFlowExecutorConfig());
        return quickFlowExecutor;
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(BusinessFlow... businessFlowList) {
        BusinessFlow firstBusinessFlow = businessFlowList[0];
        QuickFlowExecutor quickFlowExecutor = startFlow(firstBusinessFlow.name());
        for(BusinessFlow businessFlow:businessFlowList){
            quickFlowExecutor.next(businessFlow);
        }
        return quickFlowExecutor.execute();
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(CompositeBusinessFlow compositeBusinessFlow) {
        return startFlow(compositeBusinessFlow).execute();
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(FlowExecutorOption flowExecutorOption) {
        return startFlow(flowExecutorOption).execute();
    }

    /**
     * 克隆流程对象
     * */
    public QuickFlow cloneFlow(){
        return QuickFlow.newInstance()
                .putContextDataMap(flowConfig.dataMap)
                .executeGlobalTryCatchFinally(flowConfig.executeGlobalTryCatchFinally)
                .executeGlobalBeforeAfter(flowConfig.executeGlobalBeforeAfter)
                .executeGlobalSingleFlow(flowConfig.executeGlobalSingleFlow)
                .tryCatchFinallyHandler(flowConfig.tryCatchFinallyHandler)
                .beforeAfterFlowHandler(flowConfig.beforeAfterFlowHandler)
                .singleFlowListener(flowConfig.singleFlowListener);
    }

    /**
     * 获取全局配置信息
     * */
    public FlowConfig getFlowConfig(){
        return flowConfig;
    }

    /**
     * 获取流程配置项
     * */
    public JSONObject getFlowConfigRequestData(String name) {
        if(!flowConfig.flowConfigRequestDataMap.containsKey(name)){
            throw new IllegalArgumentException("流程配置数据不存在!流程名称:"+name);
        }
        JSONObject requestData = flowConfig.flowConfigRequestDataMap.get(name);
        return requestData;
    }

    /**
     * 保存流程配置项
     * */
    public QuickFlow saveFlowConfigRequestData(String name, JSONObject requestData) {
        //判断配置项值是否已存在，精确到名称+配置项key
        if(flowConfig.flowConfigRequestDataMap.containsKey(name)){
            JSONObject existRequestData = flowConfig.flowConfigRequestDataMap.get(name);
            boolean exist = true;
            for(String key:requestData.keySet()){
                if(!existRequestData.containsKey(key)){
                    existRequestData.put(key, requestData.get(key));
                    exist = false;
                }
            }
            if(exist){
                return this;
            }
        }else{
            flowConfig.flowConfigRequestDataMap.put(name, requestData);
        }
        if(null==flowConfig.flowConfigRequestDataHandler){
            throw new IllegalArgumentException("请配置FlowConfigHandler处理器!");
        }
        try {
            flowConfig.flowConfigRequestDataHandler.saveFlowConfigRequestData(name, requestData);
        }catch (Exception e){
            throw new RuntimeException(e);
        }
        return this;
    }
}
