package cn.schoolwow.quickflow.domain;

import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.QuickFlowExecutor;
import cn.schoolwow.quickflow.exception.BrokenCurrentCompositeBusinessException;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.flow.FunctionFlow;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.lang.reflect.Array;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

public class FlowContextImpl implements FlowContext {
    /**
     * 配置数据
     */
    private FlowExecutorConfig flowExecutorConfig;

    public FlowContextImpl(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    @Override
    public long getId() {
        return flowExecutorConfig.flowExecutorRootConfig.id;
    }

    @Override
    public FlowContext printTrace(boolean printTrace) {
        flowExecutorConfig.printTrace = printTrace;
        return this;
    }

    @Override
    public FlowContext ignoreBroken(boolean ignoreBroken) {
        flowExecutorConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    @Override
    public FlowContext ignoreException(boolean ignoreException) {
        flowExecutorConfig.ignoreException = ignoreException;
        return this;
    }

    @Override
    public FlowContext executeGlobalTryCatchFinally(boolean executeGlobalTryCatchFinally) {
        flowExecutorConfig.executeGlobalTryCatchFinally = executeGlobalTryCatchFinally;
        return this;
    }

    @Override
    public FlowContext executeGlobalBeforeAfter(boolean executeGlobalBeforeAfter) {
        flowExecutorConfig.executeGlobalBeforeAfter = executeGlobalBeforeAfter;
        return this;
    }

    @Override
    public FlowContext executeGlobalSingleFlow(boolean executeGlobalSingleFlow) {
        flowExecutorConfig.executeGlobalSingleFlow = executeGlobalSingleFlow;
        return this;
    }

    @Override
    public FlowContext isDataNotExist(String key, String message) {
        if(!containKey(key)){
            throw new IllegalArgumentException(message);
        }
        return this;
    }

    @Override
    public boolean containKey(String key) {
        Object value = getData(key);
        return null!=value;
    }

    @Override
    public Object checkData(String key) {
        Object value = getData(key);
        checkValueEmpty(key, value);
        return value;
    }

    @Override
    public <T> T checkData(String key, Class<T> clazz) {
        T value = getData(key, clazz);
        checkValueEmpty(key, value);
        return value;
    }

    @Override
    public Object useData(String key) {
        Object value = checkData(key);
        removeData(key);
        return value;
    }

    @Override
    public <T> T useData(String key, Class<T> clazz) {
        T value = checkData(key, clazz);
        removeData(key);
        return value;
    }

    @Override
    public Object getData(String key) {
        //首先从当前流程获取
        Map<String,Object> currentFlowDataMap = flowExecutorConfig.currentFlowDataMap;
        if(null!=currentFlowDataMap&&currentFlowDataMap.containsKey(key)){
            return flowExecutorConfig.currentFlowDataMap.get(key);
        }
        //其次从当前复合流程获取
        Map<String,Object> currentCompositeFlowDataMap = flowExecutorConfig.currentCompositeFlowDataMap.get();
        if(null!=currentCompositeFlowDataMap&&currentCompositeFlowDataMap.containsKey(key)){
            return flowExecutorConfig.currentCompositeFlowDataMap.get().get(key);
        }
        //然后从线程上下文中获取
        Map threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        if(null!=threadLocalMap&&threadLocalMap.containsKey(key)){
            return threadLocalMap.get(key);
        }
        //然后从流程上下文获取
        if(flowExecutorConfig.flowExecutorRootConfig.dataMap.containsKey(key)){
            return flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key);
        }
        //最后从全局流程上下文获取
        if(flowExecutorConfig.flowExecutorRootConfig.flowConfig.dataMap.containsKey(key)){
            return flowExecutorConfig.flowExecutorRootConfig.flowConfig.dataMap.get(key);
        }
        return null;
    }

    @Override
    public <T> T getData(String key, Class<T> clazz) {
        Object value = getData(key);
        if(value instanceof JSONObject){
            JSONObject o = (JSONObject) value;
            return o.toJavaObject(clazz);
        }
        if(value instanceof JSONArray){
            JSONArray o = (JSONArray) value;
            return o.toJavaObject(clazz);
        }
        try {
            T t = clazz.cast(value);
            return t;
        }catch (Exception e){
            throw new UnsupportedOperationException("无法转换类型!预期类型:"+clazz.getName()+",实际类型:"+value.getClass().getName());
        }
    }

    @Override
    public Object getData(String key, Object defaultValue) {
        Object value = getData(key);
        if(null==value){
            putThreadLocalData(key, defaultValue);
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public <T> T getData(String key, Class<T> clazz, T defaultValue) {
        T value = getData(key, clazz);
        if(null==value){
            putThreadLocalData(key, defaultValue);
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public Map<String, Object> getData() {
        Map<String,Object> dataMap = flowExecutorConfig.flowExecutorRootConfig.dataMap;
        Map<String,Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        if(null!=threadLocalMap){
            dataMap.putAll(threadLocalMap);
        }
        dataMap.putAll(flowExecutorConfig.flowExecutorRootConfig.flowConfig.dataMap);
        return dataMap;
    }

    @Override
    public Map<String, Object> getTemporaryDataMap() {
        return flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap;
    }

    @Override
    public Map<String, Object> getThreadLocalDataMap() {
        Map threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        return threadLocalMap;
    }

    @Override
    public Map<String, Object> getRequestDataMap() {
        return flowExecutorConfig.flowExecutorRootConfig.requestKeyMap;
    }

    @Override
    public Map<String, Object> getContextDataMap() {
        Set<String> keySet = new HashSet<>(flowExecutorConfig.flowExecutorRootConfig.dataMap.keySet());
        keySet.removeAll(flowExecutorConfig.flowExecutorRootConfig.requestKeySet);
        keySet.removeAll(flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap.keySet());
        Map<String,Object> contextDataMap = new HashMap<>();
        for(String key:keySet){
            contextDataMap.put(key, flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key));
        }
        contextDataMap.putAll(flowExecutorConfig.flowExecutorRootConfig.userDataMap);
        return contextDataMap;
    }

    @Override
    public FlowContext putContextData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorConfig.flowExecutorRootConfig.flowConfig.dataMap.put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorConfig.flowExecutorRootConfig.dataMap.put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putFunctionFlowIfAbsent(String key, FunctionFlow functionFlow) {
        if(!containKey(key)){
            putFunctionFlow(key, functionFlow);
        }
        return this;
    }

    @Override
    public FlowContext putFunctionFlow(String key, FunctionFlow functionFlow) {
        flowExecutorConfig.flowContext.putTemporaryData(key, functionFlow);
        return this;
    }

    @Override
    public FlowContext putTemporaryDataIfAbsent(String key, Object value) {
        if(!containKey(key)){
            putTemporaryData(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putTemporaryData(String key, Object value) {
        putData(key, value);
        if(null!=key){
            flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap.put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putTemporaryData(Map<String, Object> dataMap) {
        for(Map.Entry<String,Object> entry:dataMap.entrySet()){
            putTemporaryData(entry.getKey(), entry.getValue());
        }
        return this;
    }

    @Override
    public FlowContext putCurrentCompositeFlowData(String key, Object value) {
        if(null!=key){
            flowExecutorConfig.currentCompositeFlowDataMap.get().put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putCurrentFlowData(String key, Object value) {
        if(null!=key){
            flowExecutorConfig.currentFlowDataMap.put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext putThreadLocalData(String key, Object value) {
        if(null!=key&&null!=value){
            Map<String, Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
            if(null==threadLocalMap){
                threadLocalMap = new HashMap<>();
                flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.set(threadLocalMap);
            }
            threadLocalMap.put(key, value);
        }
        return this;
    }

    @Override
    public FlowContext removeData(String key) {
        if(null!=key){
            flowExecutorConfig.flowExecutorRootConfig.userDataMap.put(key, flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key));
            flowExecutorConfig.flowExecutorRootConfig.dataMap.remove(key);
        }
        return this;
    }

    @Override
    public FlowContext removeThreadLocalData(String key) {
        if(null!=key){
            Map<String,Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
            if(null!=threadLocalMap){
                threadLocalMap.remove(key);
            }
        }
        return this;
    }

    @Override
    public FlowContext remark(String remark) {
        flowExecutorConfig.getRemarkBuilder().append("|"+remark);
        return this;
    }

    @Override
    public void brokenCurrentFlow(String reason) {
        throw new BrokenCurrentFlowException(reason);
    }

    @Override
    public void brokenCurrentCompositeBusiness(String reason) {
        throw new BrokenCurrentCompositeBusinessException(reason);
    }

    @Override
    public void broken(String reason) {
        flowExecutorConfig.flowExecutorRootConfig.brokenReason = reason;
        throw new BrokenException(reason);
    }

    @Override
    public String getFlowName() {
        return flowExecutorConfig.name;
    }

    @Override
    public List<String> getFlowNameList() {
        return flowExecutorConfig.flowNameList;
    }

    @Override
    public String getFlowTrace() {
        return getFlowExecutorRootConfig().printTraceBuilder.toString();
    }

    @Override
    public Exception getFlowException() {
        return flowExecutorConfig.flowExecutorRootConfig.exception;
    }

    @Override
    public String getBrokenReason() {
        return flowExecutorConfig.flowExecutorRootConfig.brokenReason;
    }

    @Override
    public QuickFlowExecutor startFlow(String name) {
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;
        flowExecutorConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        Map<String,Object> currentCompositeFlowDataMap = new HashMap<String,Object>();
        if(null!=this.flowExecutorConfig.currentCompositeFlowDataMap.get()){
            currentCompositeFlowDataMap.putAll(this.flowExecutorConfig.currentCompositeFlowDataMap.get());
        }
        flowExecutorConfig.currentCompositeFlowDataMap.set(currentCompositeFlowDataMap);
        flowExecutorConfig.parentFlowExecutorConfig = this.flowExecutorConfig;
        flowExecutorConfig.flowExecutorRootConfig = this.flowExecutorConfig.flowExecutorRootConfig;
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    @Override
    public QuickFlowExecutor startFlow(BusinessFlow businessFlow) {
        return startFlow(businessFlow.name()).next(businessFlow);
    }

    @Override
    public QuickFlowExecutor startFlow(CompositeBusinessFlow compositeBusinessFlow) {
        return QuickFlowUtil.getCompositeQuickFlowExecutor(compositeBusinessFlow, flowExecutorConfig);
    }

    @Override
    public FlowContext executeFunctionFlowList(String... functionFlowNames) {
        for(String functionFlowName:functionFlowNames){
            FunctionFlow functionFlow = (FunctionFlow) checkData(functionFlowName);
            try {
                functionFlow.executeFunction();
            } catch (BrokenException e){
                throw e;
            } catch (Exception e){
                throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
            }
        }
        return this;
    }

    @Override
    public FlowContext executeFlowList(BusinessFlow... businessFlowList) {
        Boolean printTrace = flowExecutorConfig.printTrace;
        try {
            flowExecutorConfig.printTrace = false;
            for(BusinessFlow businessFlow:businessFlowList){
                try {
                    businessFlow.executeBusinessFlow(this);
                }catch (BrokenCurrentFlowException e){
                    flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+ businessFlow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
                }
            }
        }catch (BrokenException e){
            throw e;
        }catch (Exception e){
            throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
        }finally {
            flowExecutorConfig.printTrace = printTrace;
        }
        return this;
    }

    @Override
    public FlowContext executeCompositeFlowList(CompositeBusinessFlow... compositeBusinessFlowList) {
        for(CompositeBusinessFlow compositeBusinessFlow:compositeBusinessFlowList){
            startFlow(compositeBusinessFlow).executeGlobalTryCatchFinally(false).execute();
        }
        return this;
    }

    @Override
    public FlowExecutorConfig getFlowExecutorConfig() {
        return flowExecutorConfig;
    }

    @Override
    public FlowExecutorRootConfig getFlowExecutorRootConfig() {
        return flowExecutorConfig.flowExecutorRootConfig;
    }

    @Override
    public QuickFlow getQuickFlow() {
        return flowExecutorConfig.flowExecutorRootConfig.quickFlow;
    }

    @Override
    public String getRecord() {
        return flowExecutorConfig.flowExecutorRootConfig.recordBuilder.toString();
    }

    @Override
    public String getLog() {
        StringBuilder builder = (StringBuilder) getData("__FLOWCONTEXT_LOG", new StringBuilder());
        return builder.toString();
    }

    @Override
    public FlowContext log(String logContent, Object... parameters) {
        if(null!=parameters&&parameters.length>0){
            for(Object parameter:parameters){
                logContent = logContent.replaceFirst("\\{}", parameter.toString());
            }
        }

        StringBuilder builder = (StringBuilder) getData("__FLOWCONTEXT_LOG", new StringBuilder());
        builder.append(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS"))+" "+logContent+"\r\n");
        return this;
    }

    private void checkValueEmpty(String key, Object value){
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value.getClass().isArray()){
            if(Array.getLength(value)==0){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的数组不能为空!");
            }
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
    }

}
