package cn.schoolwow.quickflow.domain;

import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.util.*;

public class FlowContextImpl implements FlowContext {
    /**
     * 配置数据
     */
    private FlowExecutorRootConfig flowExecutorRootConfig;

    public FlowContextImpl(FlowExecutorRootConfig flowExecutorRootConfig) {
        this.flowExecutorRootConfig = flowExecutorRootConfig;
    }

    @Override
    public Object checkData(String key) {
        Object value = getData(key);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public <T> T checkData(String key, Class<T> clazz) {
        T value = getData(key, clazz);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public Object getData(String key) {
        Object value = flowExecutorRootConfig.dataMap.get(key);
        if(null!=value){
            return value;
        }
        Map threadLocalMap = flowExecutorRootConfig.threadLocalDataMap.get();
        if(null==threadLocalMap){
            return null;
        }
        return threadLocalMap.get(key);
    }

    @Override
    public <T> T getData(String key, Class<T> clazz) {
        Object value = getData(key);
        if(value instanceof JSONObject){
            JSONObject o = (JSONObject) value;
            return o.toJavaObject(clazz);
        }
        if(value instanceof JSONArray){
            JSONArray o = (JSONArray) value;
            return o.toJavaObject(clazz);
        }
        try {
            T t = clazz.cast(value);
            return t;
        }catch (Exception e){
            throw new UnsupportedOperationException("无法转换类型!预期类型:"+clazz.getName()+",实际类型:"+value.getClass().getName());
        }
    }

    @Override
    public Object getData(String key, Object defaultValue) {
        Object value = getData(key);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public <T> T getData(String key, Class<T> clazz, T defaultValue) {
        T value = getData(key, clazz);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public Map<String, Object> getData() {
        Map<String,Object> dataMap = flowExecutorRootConfig.dataMap;
        Map<String,Object> threadLocalMap = flowExecutorRootConfig.threadLocalDataMap.get();
        if(null!=threadLocalMap){
            dataMap.putAll(threadLocalMap);
        }
        return dataMap;
    }

    @Override
    public void putData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorRootConfig.dataMap.put(key, value);
        }
    }

    @Override
    public void putData(Map<String, Object> map) {
        if(null!=map){
            Iterator<Map.Entry<String,Object>> iterator = map.entrySet().iterator();
            while(iterator.hasNext()){
                if(null==iterator.next().getValue()){
                    iterator.remove();
                }
            }
            flowExecutorRootConfig.dataMap.putAll(map);
        }
    }

    @Override
    public Map<String, Object> getTemporaryDataMap() {
        Map<String,Object> temporaryDataMap = new HashMap<>();
        for(String key:flowExecutorRootConfig.temporaryKeySet){
            temporaryDataMap.put(key, flowExecutorRootConfig.dataMap.get(key));
        }
        return temporaryDataMap;
    }

    @Override
    public Map<String, Object> getThreadLocalDataMap() {
        Map threadLocalMap = flowExecutorRootConfig.threadLocalDataMap.get();
        return threadLocalMap;
    }

    @Override
    public Map<String, Object> getRequestDataMap() {
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String key:flowExecutorRootConfig.requestKeySet){
            requestDataMap.put(key, flowExecutorRootConfig.dataMap.get(key));
        }
        return requestDataMap;
    }

    @Override
    public Map<String, Object> getContextDataMap() {
        Set<String> keySet = new HashSet<>(flowExecutorRootConfig.dataMap.keySet());
        keySet.removeAll(flowExecutorRootConfig.requestKeySet);
        keySet.removeAll(flowExecutorRootConfig.temporaryKeySet);
        Map<String,Object> contextDataMap = new HashMap<>();
        for(String key:keySet){
            contextDataMap.put(key, flowExecutorRootConfig.dataMap.get(key));
        }
        return contextDataMap;
    }

    @Override
    public void putTemporaryData(String key, Object value) {
        putData(key, value);
        if(null!=key){
            flowExecutorRootConfig.temporaryKeySet.add(key);
        }
    }

    @Override
    public void putTemporaryData(Map<String, Object> map) {
        putData(map);
        if(null!=map){
            flowExecutorRootConfig.temporaryKeySet.addAll(map.keySet());
        }
    }

    @Override
    public void putThreadLocalData(String key, Object value) {
        if(null!=key&&null!=value){
            Map<String, Object> threadLocalMap = flowExecutorRootConfig.threadLocalDataMap.get();
            if(null==threadLocalMap){
                threadLocalMap = new HashMap<>();
                flowExecutorRootConfig.threadLocalDataMap.set(threadLocalMap);
            }
            threadLocalMap.put(key, value);
        }
    }

    @Override
    public void putThreadLocalData(Map<String, Object> map) {
        if(null!=map){
            Map<String, Object> threadLocalMap = flowExecutorRootConfig.threadLocalDataMap.get();
            if(null==threadLocalMap){
                threadLocalMap = new HashMap<>();
                flowExecutorRootConfig.threadLocalDataMap.set(threadLocalMap);
            }
            threadLocalMap.putAll(map);
        }
    }

    @Override
    public void removeData(String key) {
        if(null!=key){
            flowExecutorRootConfig.dataMap.remove(key);
        }
    }

    @Override
    public void remark(String remark) {
        flowExecutorRootConfig.remarkBuilder.append("|"+remark);
    }

    @Override
    public void brokenCurrentFlow(String reason) {
        throw new BrokenCurrentFlowException(reason);
    }

    @Override
    public void broken(String reason) {
        throw new BrokenException(reason);
    }

    @Override
    public String getFlowName() {
        return flowExecutorRootConfig.name;
    }

    @Override
    public List<String> getFlowNameList() {
        return flowExecutorRootConfig.flowNameList;
    }

    @Override
    public String getFlowTrace() {
        return flowExecutorRootConfig.printTraceBuilder.toString();
    }

    @Override
    public Exception getFlowException() {
        return flowExecutorRootConfig.exception;
    }

    @Override
    public String getBrokenReason() {
        return flowExecutorRootConfig.brokenReason;
    }

    @Override
    public FlowContext executeFlowList(BusinessFlow... businessFlowList) throws Exception {
        return executeFlowList(new ExecuteFlowListOption(), businessFlowList);
    }

    @Override
    public FlowContext executeFlowList(String name, BusinessFlow... businessFlowList) throws Exception {
        return executeFlowList(ExecuteFlowListOptionBuilder.get().name(name).build(), businessFlowList);
    }

    @Override
    public FlowContext executeFlowList(ExecuteFlowListOption executeFlowListOption, BusinessFlow... businessFlowList) throws Exception {
        if(null!=executeFlowListOption&&executeFlowListOption.printTrace&&null!=executeFlowListOption.name){
            flowExecutorRootConfig.printFlowTrace("["+executeFlowListOption.name+"]");
        }
        flowExecutorRootConfig.incrementFlowLevel();
        BusinessFlow currentFlow = null;
        try {
            for(BusinessFlow businessFlow:businessFlowList){
                currentFlow = businessFlow;
                long startTime = System.currentTimeMillis();
                boolean hasBrokenCurrentFlowException = false;
                try {
                    businessFlow.executeBusinessFlow(this);
                } catch (BrokenCurrentFlowException e){
                    hasBrokenCurrentFlowException = true;
                    long endTime = System.currentTimeMillis();
                    if(null!=executeFlowListOption&&executeFlowListOption.printTrace){
                        flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]"+(flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "")+"|[x]|"+ businessFlow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorRootConfig.remarkBuilder.toString());
                    }
                } finally {
                    long endTime = System.currentTimeMillis();
                    if(!hasBrokenCurrentFlowException&&null!=executeFlowListOption&&executeFlowListOption.printTrace){
                        flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]"+(flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + businessFlow.name() + flowExecutorRootConfig.remarkBuilder.toString());
                    }
                    flowExecutorRootConfig.remarkBuilder.setLength(0);
                }
            }
        }catch (BrokenException e){
            if(null!=executeFlowListOption&&executeFlowListOption.printTrace){
                flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]|[x]|"+currentFlow.name()+"|业务流程终止|原因:"+e.getReason());
            }
        }catch (Exception e){
            if(null!=executeFlowListOption&&executeFlowListOption.printTrace){
                flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]|[x]|"+currentFlow.name()+"|业务流程发生异常|异常信息:" + e.getClass().getName() + "-" + e.getMessage());
            }
            if(null!=executeFlowListOption&&executeFlowListOption.ignoreException){
                e.printStackTrace();
            }else{
                throw e;
            }
        }finally {
            flowExecutorRootConfig.decrementFlowLevel();
            if(null!=executeFlowListOption&&executeFlowListOption.printTrace&&null!=executeFlowListOption.name){
                flowExecutorRootConfig.printFlowTrace("["+executeFlowListOption.name+"]");
            }
        }
        return this;
    }

    @Override
    public BusinessFlow getBusinessFlow(String flowName) {
        BusinessFlow businessFlow = flowExecutorRootConfig.flowConfig.businessMap.get(flowName);
        if(null==businessFlow){
            throw new IllegalArgumentException("流程实例不存在!请先调用save方法保存该流程!流程名称:"+flowName);
        }
        return businessFlow;
    }

}
