package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.*;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.FlowNameInstanceMappingHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;

public class QuickFlow {
    private Logger logger = LoggerFactory.getLogger(QuickFlow.class);

    /**流程配置信息*/
    private FlowConfig flowConfig = new FlowConfig();

    /**默认流程配置对象*/
    private static QuickFlow defaultQuickFlow = new QuickFlow();

    public static QuickFlow newInstance() {
        return new QuickFlow();
    }

    /**获取默认QuickFlow对象*/
    public static QuickFlow defaultQuickFlow(){
        return defaultQuickFlow;
    }

    /**执行静态业务逻辑(仅执行一次)*/
    public static QuickFlow executeStaticFlowList(BusinessFlow... businessFlows){
        defaultQuickFlow.executeFlow(businessFlows);
        return defaultQuickFlow();
    }

    /**
     * 是否打印执行路径,默认为true
     */
    public QuickFlow printTrace(boolean printTrace) {
        flowConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略中断,默认为true
     */
    public QuickFlow ignoreBroken(boolean ignoreBroken) {
        flowConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    /**
     * 是否忽略异常,默认为false
     */
    public QuickFlow ignoreException(boolean ignoreException) {
        flowConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 设置全局上下文参数
     */
    public QuickFlow putContextData(String key, Object value) {
        flowConfig.dataMap.put(key, value);
        return this;
    }

    /**
     * 设置全局上下文参数
     */
    public QuickFlow putContextData(Map<String,Object> dataMap) {
        flowConfig.dataMap.putAll(dataMap);
        return this;
    }

    /**
     * 是否所有流程都执行全局tryCatchFinally处理器
     */
    public QuickFlow executeGlobalTryCatchFinally(boolean executeGlobalTryCatchFinally) {
        flowConfig.executeGlobalTryCatchFinally = executeGlobalTryCatchFinally;
        return this;
    }

    /**
     * 是否所有流程都执行全局beforeAfter处理器
     */
    public QuickFlow executeGlobalBeforeAfter(boolean executeGlobalBeforeAfter) {
        flowConfig.executeGlobalBeforeAfter = executeGlobalBeforeAfter;
        return this;
    }

    /**
     * 是否所有流程都执行全局singleFlow处理器
     */
    public QuickFlow executeGlobalSingleFlow(boolean executeGlobalSingleFlow) {
        flowConfig.executeGlobalSingleFlow = executeGlobalSingleFlow;
        return this;
    }

    /**
     * 指定流程名称和实例映射
     */
    public QuickFlow flowNameInstanceMappingHandler(FlowNameInstanceMappingHandler flowNameInstanceMappingHandler) {
        flowConfig.flowNameInstanceMappingHandler = flowNameInstanceMappingHandler;
        return this;
    }

    /**
     * 流程执行前后事件监听
     */
    public QuickFlow singleFlowListener(SingleFlowListener singleFlowListener) {
        flowConfig.singleFlowListener = singleFlowListener;
        return this;
    }

    /**
     * 主流程列表前后添加流程列表
     */
    public QuickFlow beforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        flowConfig.beforeAfterFlowHandler = beforeAfterFlowHandler;
        return this;
    }

    /**
     * 指定try-catch-finally流程列表
     */
    public QuickFlow tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(String name) {
        if(null==name||name.isEmpty()){
            throw new IllegalArgumentException("请指定主流程名称!");
        }
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;

        FlowExecutorRootConfig flowExecutorRootConfig = new FlowExecutorRootConfig();
        flowExecutorRootConfig.threadLocalDataMap.set(new HashMap<>());
        flowExecutorRootConfig.flowConfig = flowConfig;
        flowExecutorRootConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        flowExecutorRootConfig.quickFlow = this;

        flowExecutorConfig.flowExecutorRootConfig = flowExecutorRootConfig;
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(BusinessFlow businessFlow) {
        return startFlow(businessFlow.name())
                .next(businessFlow);
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(FlowExecutorOption flowExecutorOption) {
        if(null==flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        if(null==flowExecutorOption.flowNameList||flowExecutorOption.flowNameList.isEmpty()){
            throw new IllegalArgumentException("请指定流程列表!");
        }
        QuickFlowExecutor quickFlowExecutor = startFlow(flowExecutorOption.name)
                .putData(flowExecutorOption.requestData);
        for(String flowName:flowExecutorOption.flowNameList){
            if(flowName.startsWith("handler-")){
                int indexOf = flowName.indexOf(":")+1;
                String handlerName = flowName.substring(indexOf);
                if(flowName.startsWith("handler-tryCatchFinally:")){
                    TryCatchFinallyHandler tryCatchFinallyHandler = flowConfig.flowNameInstanceMappingHandler.getTryCatchFinallyHandlerByName(handlerName);
                    if(null==tryCatchFinallyHandler){
                        throw new IllegalArgumentException("不存在tryCatchFinally处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.tryCatchFinallyHandler(tryCatchFinallyHandler);
                }
                if(flowName.startsWith("handler-beforeAfter:")){
                    BeforeAfterFlowHandler beforeAfterFlowHandler = flowConfig.flowNameInstanceMappingHandler.getBeforeAfterHandlerByName(flowName.substring(indexOf));
                    if(null==beforeAfterFlowHandler){
                        throw new IllegalArgumentException("不存在beforeAfter处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.beforeAfterFlowHandler(beforeAfterFlowHandler);
                }
                if(flowName.startsWith("handler-singleFlow:")){
                    SingleFlowListener singleFlowListener = flowConfig.flowNameInstanceMappingHandler.getSingleFlowListener(flowName.substring(indexOf));
                    if(null==singleFlowListener){
                        throw new IllegalArgumentException("不存在singleFlow处理器!处理器名称:"+handlerName);
                    }
                    quickFlowExecutor.singleFlowListener(singleFlowListener);
                }
            } else if(flowName.startsWith("composite:")){
                String compositeFlowName = flowName.substring("composite:".length());
                CompositeBusinessFlow compositeBusinessFlow = flowConfig.flowNameInstanceMappingHandler.getCompositeFlowByName(compositeFlowName);
                if(null==compositeBusinessFlow){
                    throw new IllegalArgumentException("不存在复合流程!复合流程名称:"+compositeFlowName);
                }
                quickFlowExecutor.next(compositeBusinessFlow);
            }else{
                BusinessFlow businessFlow = flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName);
                if(null==businessFlow){
                    throw new IllegalArgumentException("不存在流程!流程名称:"+flowName);
                }
                quickFlowExecutor.next(businessFlow);
            }
        }
        return quickFlowExecutor;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(CompositeBusinessFlow compositeBusinessFlow) {
        return compositeBusinessFlow.getCompositeBusiness();
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(BusinessFlow... businessFlowList) {
        BusinessFlow firstBusinessFlow = businessFlowList[0];
        QuickFlowExecutor quickFlowExecutor = startFlow(firstBusinessFlow.name());
        for(BusinessFlow businessFlow:businessFlowList){
            quickFlowExecutor.next(businessFlow);
        }
        return quickFlowExecutor.execute();
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(CompositeBusinessFlow compositeBusinessFlow) {
        return compositeBusinessFlow.getCompositeBusiness().execute();
    }

    /**
     * 执行流程
     * */
    public FlowContext executeFlow(FlowExecutorOption flowExecutorOption) {
        return startFlow(flowExecutorOption).execute();
    }

    /**
     * 获取全局配置信息
     * */
    public FlowConfig getFlowConfig(){
        return flowConfig;
    }
}
