package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowConfig;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class QuickFlowExecutorInner {
    private FlowExecutorConfig flowExecutorConfig;

    private FlowExecutorRootConfig flowExecutorRootConfig;

    private FlowConfig flowConfig;

    public QuickFlowExecutorInner(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
        this.flowExecutorRootConfig = flowExecutorConfig.flowExecutorRootConfig;
        this.flowConfig = this.flowExecutorRootConfig.flowConfig;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext doExecute() {
        setRequestDataMap();
        flowExecutorConfig.flowNameList.addAll(QuickFlowUtil.getFlowNameList(flowExecutorConfig));
        try {
            beforeExecuteBusiness();
            executeBusiness();
        } catch (Exception e) {
            handleBusinessException(e);
        } finally {
            handleBusinessFinally();
        }
        return flowExecutorConfig.flowExecutorRootConfig.flowContext;
    }

    /**固定请求参数*/
    private void setRequestDataMap(){
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String requestKey:flowExecutorRootConfig.requestKeySet){
            requestDataMap.put(requestKey, flowExecutorRootConfig.dataMap.get(requestKey));
        }
        flowExecutorRootConfig.requestKeyMap = new JSONObject(requestDataMap);
    }

    /**执行流程列表之前*/
    private void beforeExecuteBusiness(){
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.recordContent("当前流程是主流程!流程id:"+flowExecutorRootConfig.id);
            flowExecutorConfig.printFlowTrace("流程id:"+flowExecutorRootConfig.id);
        }
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "try", null);
        executeBeforeAfterFlowHandler(true, flowExecutorConfig.beforeAfterFlowHandler);
        {
            boolean shouldExecuteGlobalTryCatchFinally = null!=flowExecutorConfig.executeGlobalTryCatchFinally?flowExecutorConfig.executeGlobalTryCatchFinally:flowConfig.executeGlobalTryCatchFinally;
            flowExecutorConfig.recordContent("是否执行全局tryCatchFinally处理器-try:"+shouldExecuteGlobalTryCatchFinally);
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "try", null);
        }
        {
            boolean shouldExecuteGlobalBeforeAfter = null!=flowExecutorConfig.executeGlobalBeforeAfter?flowExecutorConfig.executeGlobalBeforeAfter:flowConfig.executeGlobalBeforeAfter;
            flowExecutorConfig.recordContent("是否执行全局beforeAfter处理器:"+shouldExecuteGlobalBeforeAfter);
            executeBeforeAfterFlowHandler(shouldExecuteGlobalBeforeAfter, flowConfig.beforeAfterFlowHandler);
        }
        flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        flowExecutorConfig.flowExecutorRootConfig.incrementFlowLevel();
    }

    private void executeBusiness() throws Exception {
        for (BusinessFlow flow : flowExecutorConfig.flowList) {
            flowExecutorConfig.getRemarkBuilder().setLength(0);
            flowExecutorConfig.currentFlow = flow;
            executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "before");
            {
                boolean shouldExecuteGlobalSingleFlow = null!=flowExecutorConfig.executeGlobalSingleFlow?flowExecutorConfig.executeGlobalSingleFlow:flowConfig.executeGlobalSingleFlow;
                flowExecutorConfig.recordContent("是否执行全局singleFlow-before处理器:"+shouldExecuteGlobalSingleFlow);
                executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "before");
            }
            long startTime = System.currentTimeMillis();
            try {
                flow.executeBusinessFlow(flowExecutorConfig.flowExecutorRootConfig.flowContext);
                long endTime = System.currentTimeMillis();
                if(!flow.name().startsWith("composite:")&&flowExecutorConfig.flowList.size()>1){
                    flowExecutorConfig.recordContent("当前流程不是复合流程或者流程列表有且仅有一个复合流程");
                    flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|" + flow.name() + flowExecutorConfig.getRemarkBuilder().toString());
                }else{
                    flowExecutorConfig.recordContent("当前流程是复合流程");
                }
            }catch (BrokenCurrentFlowException e){
                long endTime = System.currentTimeMillis();
                flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
            }
            executeSingleFlowHandler(true, flow, flowExecutorConfig.singleFlowListener, "after");
            {
                boolean shouldExecuteGlobalSingleFlow = null!=flowExecutorConfig.executeGlobalSingleFlow?flowExecutorConfig.executeGlobalSingleFlow:flowConfig.executeGlobalSingleFlow;
                flowExecutorConfig.recordContent("是否执行全局singleFlow-after处理器:"+shouldExecuteGlobalSingleFlow);
                executeSingleFlowHandler(shouldExecuteGlobalSingleFlow, flow, flowConfig.singleFlowListener, "after");
            }
        }
    }

    private void handleBusinessException(Exception e){
        Exception targetException = e;
        QuickFlowRuntimeException quickFlowRuntimeException = null;
        if(e instanceof QuickFlowRuntimeException){
            targetException = ((QuickFlowRuntimeException)e).targetException;
            quickFlowRuntimeException = (QuickFlowRuntimeException) e;
        }else{
            quickFlowRuntimeException = new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
        }
        //判断异常类型
        if(targetException instanceof BrokenException){
            handleBrokenException((BrokenException) targetException);
        }else{
            handleException(e, quickFlowRuntimeException);
        }
    }

    private void handleBrokenException(BrokenException e){
        boolean shouldIgnoreBroken = null!=flowExecutorConfig.ignoreBroken?flowExecutorConfig.ignoreBroken:flowExecutorConfig.flowExecutorRootConfig.flowConfig.ignoreBroken;
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程发生中断|原因:"+e.getReason()+(shouldIgnoreBroken?"|忽略该中断":"")+ flowExecutorConfig.getRemarkBuilder().toString());
        }else{
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.brokenReason = e.getReason();
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程终止|原因:"+e.getReason()+flowExecutorConfig.getRemarkBuilder().toString());
        }
        if(!shouldIgnoreBroken){
            throw e;
        }
    }

    private void handleException(Exception targetException, QuickFlowRuntimeException quickFlowRuntimeException){
        boolean shouldIgnoreException = null!=flowExecutorConfig.ignoreException?flowExecutorConfig.ignoreException:flowExecutorConfig.flowExecutorRootConfig.flowConfig.ignoreException;
        String exceptionMessage = targetException.getClass().getName() + "-" + targetException.getMessage();
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            //非主流程
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
        }else{
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(shouldIgnoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.exception = targetException;
            {
                boolean shouldExecuteGlobalTryCatchFinally = null!=flowExecutorConfig.executeGlobalTryCatchFinally?flowExecutorConfig.executeGlobalTryCatchFinally:flowConfig.executeGlobalTryCatchFinally;
                flowExecutorConfig.recordContent("是否执行全局tryCatchFinally处理器-catch:"+shouldExecuteGlobalTryCatchFinally);
                executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "catch", targetException);
            }
        }
        if(!shouldIgnoreException){
            throw quickFlowRuntimeException;
        }
    }

    private void handleBusinessFinally(){
        flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
        executeTryCatchFinallyHandler(true, flowExecutorConfig.tryCatchFinallyHandler, "finally", null);
        {
            boolean shouldExecuteGlobalTryCatchFinally = null!=flowExecutorConfig.executeGlobalTryCatchFinally?flowExecutorConfig.executeGlobalTryCatchFinally:flowConfig.executeGlobalTryCatchFinally;
            flowExecutorConfig.recordContent("是否执行全局tryCatchFinally处理器-finally:"+shouldExecuteGlobalTryCatchFinally);
            executeTryCatchFinallyHandler(shouldExecuteGlobalTryCatchFinally, flowConfig.tryCatchFinallyHandler, "finally", null);
        }
        for(String temporaryKey:flowExecutorConfig.temporaryDataKeySet){
            flowExecutorConfig.flowExecutorRootConfig.dataMap.remove(temporaryKey);
        }
        flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
    }

    private void executeTryCatchFinallyHandler(boolean shouldExecute, TryCatchFinallyHandler tryCatchFinallyHandler, String operation, Exception e) {
        if(null==tryCatchFinallyHandler||!shouldExecute){
            return;
        }
        long startTime = System.currentTimeMillis();
        try {
            switch (operation){
                case "try":{tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowExecutorRootConfig.flowContext);}break;
                case "catch":{tryCatchFinallyHandler.handleException(flowExecutorConfig.flowExecutorRootConfig.flowContext, e);}break;
                case "finally":{
                    if(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler==tryCatchFinallyHandler){
                        flowExecutorConfig.printFlowTraceWithIndex(operation+":"+tryCatchFinallyHandler.name());
                    }
                    tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowExecutorRootConfig.flowContext);
                }break;
                default:{
                    throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
                }
            }
            long endTime = System.currentTimeMillis();
            flowExecutorConfig.printFlowTraceWithIndex("|" + (endTime - startTime) + "毫秒|"+operation+":"+tryCatchFinallyHandler.name());
        }catch (Exception ex){
            if(ex instanceof QuickFlowRuntimeException){
                throw (QuickFlowRuntimeException)ex;
            }else{
                throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, ex);
            }
        }
    }

    private void executeBeforeAfterFlowHandler(boolean shouldExecute, BeforeAfterFlowHandler beforeAfterFlowHandler) {
        if(null==beforeAfterFlowHandler||!shouldExecute){
            return;
        }
        List<BusinessFlow> beforeFlowList = beforeAfterFlowHandler.getBeforeFlowList();
        if(null!=beforeFlowList&&!beforeFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(0, beforeFlowList);
        }
        List<BusinessFlow> afterFlowList = beforeAfterFlowHandler.getAfterFlowList();
        if(null!=afterFlowList&&!afterFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(afterFlowList);
        }
    }

    private void executeSingleFlowHandler(boolean shouldExecute, BusinessFlow flow, SingleFlowListener singleFlowListener, String operation) throws Exception {
        if(null==singleFlowListener||!shouldExecute){
            return;
        }
        switch (operation){
            case "before":{
                singleFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowExecutorRootConfig.flowContext);
            };break;
            case "after":{
                singleFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowExecutorRootConfig.flowContext);
            };break;
            default:{
                throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
            }
        }
    }

}
