package cn.schoolwow.quickflow.domain;

import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.QuickFlowExecutor;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.util.*;

public class FlowContextImpl implements FlowContext {
    /**
     * 配置数据
     */
    private FlowExecutorConfig flowExecutorConfig;

    public FlowContextImpl(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    @Override
    public long getId() {
        return flowExecutorConfig.flowExecutorRootConfig.id;
    }

    @Override
    public FlowContext printTrace(boolean printTrace) {
        flowExecutorConfig.printTrace = printTrace;
        return this;
    }

    @Override
    public FlowContext ignoreBroken(boolean ignoreBroken) {
        flowExecutorConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    @Override
    public FlowContext ignoreException(boolean ignoreException) {
        flowExecutorConfig.ignoreException = ignoreException;
        return this;
    }

    @Override
    public boolean containKey(String key) {
        return null!=flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key);
    }

    @Override
    public Object checkData(String key) {
        Object value = getData(key);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value.getClass().isArray()){
            Object[] array = (Object[]) value;
            if(array.length==0){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的数组不能为空!");
            }
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public <T> T checkData(String key, Class<T> clazz) {
        T value = getData(key, clazz);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(clazz.isArray()){
            Object[] array = (Object[]) value;
            if(array.length==0){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的数组不能为空!");
            }
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public Object useData(String key) {
        Object value = checkData(key);
        removeData(key);
        return value;
    }

    @Override
    public <T> T useData(String key, Class<T> clazz) {
        T value = checkData(key, clazz);
        removeData(key);
        return value;
    }

    @Override
    public Object getData(String key) {
        Object value = flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key);
        if(null!=value){
            return value;
        }
        Map threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        if(null==threadLocalMap){
            return null;
        }
        return threadLocalMap.get(key);
    }

    @Override
    public <T> T getData(String key, Class<T> clazz) {
        Object value = getData(key);
        if(value instanceof JSONObject){
            JSONObject o = (JSONObject) value;
            return o.toJavaObject(clazz);
        }
        if(value instanceof JSONArray){
            JSONArray o = (JSONArray) value;
            return o.toJavaObject(clazz);
        }
        try {
            T t = clazz.cast(value);
            return t;
        }catch (Exception e){
            throw new UnsupportedOperationException("无法转换类型!预期类型:"+clazz.getName()+",实际类型:"+value.getClass().getName());
        }
    }

    @Override
    public Object getData(String key, Object defaultValue) {
        Object value = getData(key);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public <T> T getData(String key, Class<T> clazz, T defaultValue) {
        T value = getData(key, clazz);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public Map<String, Object> getData() {
        Map<String,Object> dataMap = flowExecutorConfig.flowExecutorRootConfig.dataMap;
        Map<String,Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        if(null!=threadLocalMap){
            dataMap.putAll(threadLocalMap);
        }
        return dataMap;
    }

    @Override
    public void putData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorConfig.flowExecutorRootConfig.dataMap.put(key, value);
        }
    }

    @Override
    public void putData(Map<String, Object> map) {
        if(null!=map){
            map.entrySet().removeIf(stringObjectEntry -> null == stringObjectEntry.getValue());
            flowExecutorConfig.flowExecutorRootConfig.dataMap.putAll(map);
        }
    }

    @Override
    public Map<String, Object> getTemporaryDataMap() {
        return flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap;
    }

    @Override
    public Map<String, Object> getThreadLocalDataMap() {
        Map threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
        return threadLocalMap;
    }

    @Override
    public Map<String, Object> getRequestDataMap() {
        return flowExecutorConfig.flowExecutorRootConfig.requestKeyMap;
    }

    @Override
    public Map<String, Object> getContextDataMap() {
        Set<String> keySet = new HashSet<>(flowExecutorConfig.flowExecutorRootConfig.dataMap.keySet());
        keySet.removeAll(flowExecutorConfig.flowExecutorRootConfig.requestKeySet);
        keySet.removeAll(flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap.keySet());
        Map<String,Object> contextDataMap = new HashMap<>();
        for(String key:keySet){
            contextDataMap.put(key, flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key));
        }
        contextDataMap.putAll(flowExecutorConfig.flowExecutorRootConfig.userDataMap);
        return contextDataMap;
    }

    @Override
    public void putTemporaryData(String key, Object value) {
        putData(key, value);
        if(null!=key){
            flowExecutorConfig.temporaryDataKeySet.add(key);
            flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap.put(key, value);
        }
    }

    @Override
    public void putTemporaryData(Map<String, Object> map) {
        putData(map);
        if(null!=map){
            flowExecutorConfig.temporaryDataKeySet.addAll(map.keySet());
            flowExecutorConfig.flowExecutorRootConfig.temporaryDataMap.putAll(map);
        }
    }

    @Override
    public void putThreadLocalData(String key, Object value) {
        if(null!=key&&null!=value){
            Map<String, Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
            if(null==threadLocalMap){
                threadLocalMap = new HashMap<>();
                flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.set(threadLocalMap);
            }
            threadLocalMap.put(key, value);
        }
    }

    @Override
    public void putThreadLocalData(Map<String, Object> map) {
        if(null!=map){
            Map<String, Object> threadLocalMap = flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.get();
            if(null==threadLocalMap){
                threadLocalMap = new HashMap<>();
                flowExecutorConfig.flowExecutorRootConfig.threadLocalDataMap.set(threadLocalMap);
            }
            threadLocalMap.putAll(map);
        }
    }

    @Override
    public void removeData(String key) {
        if(null!=key){
            flowExecutorConfig.flowExecutorRootConfig.userDataMap.put(key, flowExecutorConfig.flowExecutorRootConfig.dataMap.get(key));
            flowExecutorConfig.flowExecutorRootConfig.dataMap.remove(key);
        }
    }

    @Override
    public void remark(String remark) {
        flowExecutorConfig.getRemarkBuilder().append("|"+remark);
    }

    @Override
    public void brokenCurrentFlow(String reason) {
        throw new BrokenCurrentFlowException(reason);
    }

    @Override
    public void broken(String reason) {
        throw new BrokenException(reason);
    }

    @Override
    public String getFlowName() {
        return flowExecutorConfig.name;
    }

    @Override
    public List<String> getFlowNameList() {
        return flowExecutorConfig.flowNameList;
    }

    @Override
    public String getFlowTrace() {
        return getFlowExecutorRootConfig().printTraceBuilder.toString();
    }

    @Override
    public Exception getFlowException() {
        return flowExecutorConfig.flowExecutorRootConfig.exception;
    }

    @Override
    public String getBrokenReason() {
        return flowExecutorConfig.flowExecutorRootConfig.brokenReason;
    }

    @Override
    public QuickFlowExecutor startFlow(String name) {
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;
        flowExecutorConfig.printTrace = this.flowExecutorConfig.printTrace;
        flowExecutorConfig.ignoreBroken = this.flowExecutorConfig.ignoreBroken;
        flowExecutorConfig.ignoreException = this.flowExecutorConfig.ignoreException;
        flowExecutorConfig.parentFlowExecutorConfig = this.flowExecutorConfig;
        flowExecutorConfig.flowExecutorRootConfig = this.flowExecutorConfig.flowExecutorRootConfig;
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    @Override
    public QuickFlowExecutor startFlow(BusinessFlow businessFlow) {
        return startFlow(businessFlow.name()).next(businessFlow);
    }

    @Override
    public QuickFlowExecutor startFlow(CompositeBusinessFlow compositeBusinessFlow) {
        FlowExecutorConfig compositeFlowExecutorConfig = compositeBusinessFlow.getCompositeBusiness().getFlowExecutorConfig();
        QuickFlowExecutor quickFlowExecutor = QuickFlowUtil.getCompositeExecutor(flowExecutorConfig, compositeFlowExecutorConfig);
        return quickFlowExecutor;
    }

    @Override
    public FlowContext executeFlowList(BusinessFlow... businessFlowList) {
        boolean printTrace = flowExecutorConfig.printTrace;
        try {
            flowExecutorConfig.printTrace = false;
            for(BusinessFlow businessFlow:businessFlowList){
                long startTime = System.currentTimeMillis();
                try {
                    businessFlow.executeBusinessFlow(this);
                } catch (BrokenCurrentFlowException e){
                    long endTime = System.currentTimeMillis();
                    flowExecutorConfig.printFlowTraceWithIndex((flowExecutorConfig.flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "")+"|[x]|"+ businessFlow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
                } finally {
                    long endTime = System.currentTimeMillis();
                    flowExecutorConfig.printFlowTraceWithIndex((flowExecutorConfig.flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + businessFlow.name() + flowExecutorConfig.getRemarkBuilder().toString());
                }
            }
        }catch (BrokenException e){
            throw e;
        }catch (Exception e){
            throw new QuickFlowRuntimeException(flowExecutorConfig.flowExecutorRootConfig.id, e);
        }finally {
            flowExecutorConfig.printTrace = printTrace;
        }
        return this;
    }

    @Override
    public FlowContext executeQuickFlowExecutor(QuickFlowExecutor quickFlowExecutor) {
        QuickFlowExecutor compositeExecutor = QuickFlowUtil.getCompositeExecutor(flowExecutorConfig, quickFlowExecutor.getFlowExecutorConfig());
        return compositeExecutor.executeCompositeBusiness();
    }

    @Override
    public FlowContext executeCompositeFlowList(CompositeBusinessFlow... compositeBusinessFlowList) {
        for(CompositeBusinessFlow compositeBusinessFlow:compositeBusinessFlowList){
            startFlow(compositeBusinessFlow).executeCompositeBusiness();
        }
        return this;
    }

    @Override
    public FlowExecutorConfig getFlowExecutorConfig() {
        return flowExecutorConfig;
    }

    @Override
    public FlowExecutorRootConfig getFlowExecutorRootConfig() {
        return flowExecutorConfig.flowExecutorRootConfig;
    }

    @Override
    public QuickFlow getQuickFlow() {
        return flowExecutorConfig.flowExecutorRootConfig.quickFlow;
    }

    @Override
    public String getRecord() {
        return flowExecutorConfig.flowExecutorRootConfig.recordBuilder.toString();
    }

}
