package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorOption;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.exception.QuickFlowRuntimeException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.CompositeBusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.SingleFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import cn.schoolwow.quickflow.util.QuickFlowUtil;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class QuickFlowExecutor {
    private Logger logger = LoggerFactory.getLogger(QuickFlowExecutor.class);

    /**配置信息*/
    private FlowExecutorConfig flowExecutorConfig;

    public QuickFlowExecutor(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(String key, Object value) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putData(key, value);
        flowExecutorConfig.flowExecutorRootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putData(dataMap);
        flowExecutorConfig.flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(String key, Object value) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putTemporaryData(key, value);
        flowExecutorConfig.flowExecutorRootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putTemporaryData(dataMap);
        flowExecutorConfig.flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(String key, Object value) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putThreadLocalData(key, value);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowExecutorRootConfig.flowContext.putThreadLocalData(dataMap);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(String flowName) {
        if(null==flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        BusinessFlow flow = flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName);
        if(null==flow){
            CompositeBusinessFlow compositeBusinessFlow = flowExecutorConfig.flowExecutorRootConfig.flowConfig.flowNameInstanceMappingHandler.getCompositeFlowByName(flowName);
            if(null==compositeBusinessFlow){
                throw new IllegalArgumentException("根据流程名称查找流程示例失败!流程名称:"+flowName);
            }
            next(compositeBusinessFlow);
        }else{
            flowExecutorConfig.flowList.add(flow);
        }
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(BusinessFlow flow) {
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(CompositeBusinessFlow compositeBusinessFlow) {
        FlowExecutorConfig compositeFlowExecutorConfig = compositeBusinessFlow.getCompositeBusiness().getFlowExecutorConfig();
        String compositeFlowName = "composite:"+compositeFlowExecutorConfig.name;

        BusinessFlow businessFlow = new BusinessFlow() {
            @Override
            public void executeBusinessFlow(FlowContext flowContext) throws Exception {
                QuickFlowExecutor quickFlowExecutor = QuickFlowUtil.getCompositeExecutor(flowContext.getFlowExecutorConfig(), compositeFlowExecutorConfig);
                quickFlowExecutor.execute();
            }

            @Override
            public String name() {
                return compositeFlowName;
            }
        };
        flowExecutorConfig.flowList.add(businessFlow);
        return this;
    }

    /**
     * 是否打印路径
     */
    public QuickFlowExecutor printTrace(boolean printTrace) {
        flowExecutorConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略中断
     */
    public QuickFlowExecutor ignoreBroken(boolean ignoreBroken) {
        flowExecutorConfig.ignoreBroken = ignoreBroken;
        return this;
    }

    /**
     * 是否忽略异常
     */
    public QuickFlowExecutor ignoreException(boolean ignoreException) {
        flowExecutorConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 单个流程事件监听
     */
    public QuickFlowExecutor singleFlowListener(SingleFlowListener singleFlowListener) {
        flowExecutorConfig.singleFlowListener = singleFlowListener;
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlowExecutor tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowExecutorConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 指定before-after流程列表
     */
    public QuickFlowExecutor beforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        flowExecutorConfig.beforeAfterFlowHandler = beforeAfterFlowHandler;
        return this;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext execute() {
        flowExecutorConfig.executeGlobalEvent = null==flowExecutorConfig.parentFlowExecutorConfig;
        return doExecute();
    }

    /**
     * 执行复合流程业务
     * 该方法不会执行全局事件
     */
    public FlowContext executeCompositeBusiness() {
        flowExecutorConfig.executeGlobalEvent = false;
        return doExecute();
    }

    /**保存流程实例*/
    public FlowExecutorOption getOption(){
        FlowExecutorOption flowExecutorOption = new FlowExecutorOption();
        flowExecutorOption.name = flowExecutorConfig.name;
        flowExecutorOption.requestData = flowExecutorConfig.flowExecutorRootConfig.dataMap;
        flowExecutorOption.flowNameList = getFlowNameList();
        return flowExecutorOption;
    }

    public FlowExecutorConfig getFlowExecutorConfig() {
        return flowExecutorConfig;
    }

    public FlowExecutorRootConfig getFlowExecutorRootConfig() {
        return flowExecutorConfig.flowExecutorRootConfig;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext doExecute() {
        //固定请求参数
        setRequestDataMap();
        //添加流程列表
        flowExecutorConfig.flowNameList.addAll(getFlowNameList());
        try {
            beforeExecuteBusiness();
            executeBusiness();
        } catch (Exception e) {
            handleBusinessException(e);
        } finally {
            handleBusinessFinally();
        }
        return flowExecutorConfig.flowExecutorRootConfig.flowContext;
    }

    /**固定请求参数*/
    private void setRequestDataMap(){
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String requestKey:flowExecutorConfig.flowExecutorRootConfig.requestKeySet){
            requestDataMap.put(requestKey, flowExecutorConfig.flowExecutorRootConfig.dataMap.get(requestKey));
        }
        flowExecutorConfig.flowExecutorRootConfig.requestKeyMap = new JSONObject(requestDataMap);
    }

    /**获取流程列表名称*/
    private List<String> getFlowNameList(){
        List<String> flowNameList = new ArrayList<>();
        if(null!=flowExecutorConfig.tryCatchFinallyHandler){
            flowNameList.add("handler-tryCatchFinally:"+flowExecutorConfig.tryCatchFinallyHandler.name());
        }
        if(null!=flowExecutorConfig.beforeAfterFlowHandler){
            flowNameList.add("handler-beforeAfter:"+flowExecutorConfig.beforeAfterFlowHandler.name());
        }
        for (BusinessFlow flow : flowExecutorConfig.flowList) {
            flowNameList.add(flow.name());
        }
        return flowNameList;
    }

    private void beforeExecuteBusiness(){
        executeTryCatchFinallyHandler(flowExecutorConfig.tryCatchFinallyHandler, "try", null);
        executeBeforeAfterFlowHandler(flowExecutorConfig.beforeAfterFlowHandler);
        if(flowExecutorConfig.executeGlobalEvent){
            executeTryCatchFinallyHandler(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "try", null);
            executeBeforeAfterFlowHandler(flowExecutorConfig.flowExecutorRootConfig.flowConfig.beforeAfterFlowHandler);
        }
        flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
        flowExecutorConfig.flowExecutorRootConfig.incrementFlowLevel();
    }

    private void executeBusiness() throws Exception {
        for (BusinessFlow flow : flowExecutorConfig.flowList) {
            flowExecutorConfig.currentFlow = flow;
            executeSingleFlowHandler(flow, flowExecutorConfig.singleFlowListener, "before");
            if(flowExecutorConfig.executeGlobalEvent){
                executeSingleFlowHandler(flow, flowExecutorConfig.flowExecutorRootConfig.flowConfig.singleFlowListener, "before");
            }
            long startTime = System.currentTimeMillis();
            try {
                flowExecutorConfig.getRemarkBuilder().setLength(0);
                flow.executeBusinessFlow(flowExecutorConfig.flowExecutorRootConfig.flowContext);
                long endTime = System.currentTimeMillis();
                if(!flow.name().startsWith("composite:")&&flowExecutorConfig.flowList.size()>1){
                    flowExecutorConfig.printFlowTraceWithIndex((flowExecutorConfig.flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + flow.name() + flowExecutorConfig.getRemarkBuilder().toString());
                }
            }catch (BrokenCurrentFlowException e){
                long endTime = System.currentTimeMillis();
                flowExecutorConfig.printFlowTraceWithIndex((flowExecutorConfig.flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "")+"|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.getRemarkBuilder().toString());
            }
            executeSingleFlowHandler(flow, flowExecutorConfig.singleFlowListener, "after");
            if(flowExecutorConfig.executeGlobalEvent){
                executeSingleFlowHandler(flow, flowExecutorConfig.flowExecutorRootConfig.flowConfig.singleFlowListener, "after");
            }
        }
    }

    private void handleBusinessException(Exception e){
        Exception targetException = e;
        QuickFlowRuntimeException quickFlowRuntimeException = null;
        if(e instanceof QuickFlowRuntimeException){
            targetException = ((QuickFlowRuntimeException)e).targetException;
            quickFlowRuntimeException = (QuickFlowRuntimeException) e;
        }else{
            quickFlowRuntimeException = new QuickFlowRuntimeException(e);
        }
        //判断异常类型
        if(targetException instanceof BrokenException){
            handleBrokenException((BrokenException) targetException);
        }else{
            handleException(e, quickFlowRuntimeException);
        }
    }

    private void handleBusinessFinally(){
        executeTryCatchFinallyHandler(flowExecutorConfig.tryCatchFinallyHandler, "finally", null);
        if(flowExecutorConfig.executeGlobalEvent){
            executeTryCatchFinallyHandler(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "finally", null);
        }
        for(String temporaryKey:flowExecutorConfig.temporaryDataKeySet){
            flowExecutorConfig.flowExecutorRootConfig.dataMap.remove(temporaryKey);
        }
        flowExecutorConfig.flowExecutorRootConfig.decrementFlowLevel();
        flowExecutorConfig.printFlowTrace("["+flowExecutorConfig.name+"]");
    }

    /**执行try-catch-finally事件*/
    private void executeTryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler, String operation, Exception e) {
        if(null==tryCatchFinallyHandler){
            return;
        }
        long startTime = System.currentTimeMillis();
        try {
            switch (operation){
                case "try":{tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowExecutorRootConfig.flowContext);}break;
                case "catch":{tryCatchFinallyHandler.handleException(flowExecutorConfig.flowExecutorRootConfig.flowContext, e);}break;
                case "finally":{tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowExecutorRootConfig.flowContext);}break;
                default:{
                    throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
                }
            }
            long endTime = System.currentTimeMillis();
            flowExecutorConfig.printFlowTraceWithIndex((flowExecutorConfig.flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "|")+operation+":"+tryCatchFinallyHandler.name());
        }catch (Exception ex){
            if(ex instanceof QuickFlowRuntimeException){
                throw (QuickFlowRuntimeException)ex;
            }else{
                throw new QuickFlowRuntimeException(ex);
            }
        }
    }

    private void executeBeforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        if(null==beforeAfterFlowHandler){
            return;
        }
        List<BusinessFlow> beforeFlowList = beforeAfterFlowHandler.getBeforeFlowList();
        if(null!=beforeFlowList&&!beforeFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(0, beforeFlowList);
        }
        List<BusinessFlow> afterFlowList = beforeAfterFlowHandler.getAfterFlowList();
        if(null!=afterFlowList&&!afterFlowList.isEmpty()){
            this.flowExecutorConfig.flowList.addAll(afterFlowList);
        }
    }

    private void executeSingleFlowHandler(BusinessFlow flow, SingleFlowListener singleFlowListener, String operation) throws Exception {
        if(null==singleFlowListener){
            return;
        }
        switch (operation){
            case "before":{
                singleFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowExecutorRootConfig.flowContext);
            };break;
            case "after":{
                singleFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowExecutorRootConfig.flowContext);
            };break;
            default:{
                throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
            }
        }
    }

    private void handleBrokenException(BrokenException e){
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法") + "|业务流程发生中断"+(flowExecutorConfig.ignoreBroken?"|忽略该中断":"")+ flowExecutorConfig.getRemarkBuilder().toString());
        }else{
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.brokenReason = e.getReason();
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件执行中断方法")+"|业务流程终止|原因:"+e.getReason()+flowExecutorConfig.getRemarkBuilder().toString());
        }
        boolean ignoreBroken = flowExecutorConfig.ignoreBroken;
        if(!ignoreBroken){
            throw e;
        }
    }

    private void handleException(Exception targetException, QuickFlowRuntimeException quickFlowRuntimeException){
        String exceptionMessage = targetException.getClass().getName() + "-" + targetException.getMessage();
        executeTryCatchFinallyHandler(flowExecutorConfig.tryCatchFinallyHandler, "catch", targetException);
        if(null!=flowExecutorConfig.parentFlowExecutorConfig){
            //非主流程
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(flowExecutorConfig.ignoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
        }else{
            flowExecutorConfig.printFlowTraceWithIndex("|[x]|"+(null!=flowExecutorConfig.currentFlow?flowExecutorConfig.currentFlow.name():"事件抛出异常")+(flowExecutorConfig.ignoreException?"|忽略该异常":"")+"|异常信息:" + exceptionMessage);
            //主流程
            flowExecutorConfig.flowExecutorRootConfig.exception = targetException;
            if(flowExecutorConfig.executeGlobalEvent){
                executeTryCatchFinallyHandler(flowExecutorConfig.flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "catch", targetException);
            }
        }
        boolean ignoreException = flowExecutorConfig.ignoreException;
        if(!ignoreException){
            throw quickFlowRuntimeException;
        }
    }
}
