package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorOption;
import cn.schoolwow.quickflow.domain.FlowExecutorRootConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.BeforeAfterFlowHandler;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class QuickFlowExecutor {
    private Logger logger = LoggerFactory.getLogger(QuickFlowExecutor.class);

    /**配置信息*/
    private FlowExecutorRootConfig flowExecutorRootConfig;

    public QuickFlowExecutor(FlowExecutorRootConfig flowExecutorRootConfig) {
        this.flowExecutorRootConfig = flowExecutorRootConfig;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(String key, Object value) {
        flowExecutorRootConfig.flowContext.putData(key, value);
        flowExecutorRootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(Map<String,Object> dataMap) {
        flowExecutorRootConfig.flowContext.putData(dataMap);
        flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(String key, Object value) {
        flowExecutorRootConfig.flowContext.putTemporaryData(key, value);
        flowExecutorRootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(Map<String,Object> dataMap) {
        flowExecutorRootConfig.flowContext.putTemporaryData(dataMap);
        flowExecutorRootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(String key, Object value) {
        flowExecutorRootConfig.flowContext.putThreadLocalData(key, value);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(Map<String,Object> dataMap) {
        flowExecutorRootConfig.flowContext.putThreadLocalData(dataMap);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(String flowName) {
        if(null==flowExecutorRootConfig.flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        BusinessFlow flow = flowExecutorRootConfig.flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName);
        flowExecutorRootConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(BusinessFlow flow) {
        flowExecutorRootConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlowExecutor tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowExecutorRootConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 指定before-after流程列表
     */
    public QuickFlowExecutor beforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) {
        flowExecutorRootConfig.beforeAfterFlowHandler = beforeAfterFlowHandler;
        return this;
    }

    /**
     * 是否打印路径
     */
    public QuickFlowExecutor printTrace(boolean printTrace) {
        flowExecutorRootConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略异常
     */
    public QuickFlowExecutor ignoreException(boolean ignoreException) {
        flowExecutorRootConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext execute() throws Exception {
        if (flowExecutorRootConfig.printTrace&&flowExecutorRootConfig.flowConfig.printTrace) {
            flowExecutorRootConfig.flowNameList.addAll(getFlowNameList());
        }
        BusinessFlow currentFlow = flowExecutorRootConfig.flowList.get(0);
        try {
            executeTryCatchFinallyHandler(flowExecutorRootConfig.tryCatchFinallyHandler, "try", null);
            executeTryCatchFinallyHandler(flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "try", null);
            executeBeforeAfterFlowHandler(flowExecutorRootConfig.beforeAfterFlowHandler);
            executeBeforeAfterFlowHandler(flowExecutorRootConfig.flowConfig.beforeAfterFlowHandler);
            flowExecutorRootConfig.printFlowTrace("["+flowExecutorRootConfig.name+"]");
            flowExecutorRootConfig.incrementFlowLevel();
            //执行任务
            for (BusinessFlow flow : flowExecutorRootConfig.flowList) {
                currentFlow = flow;
                if (null != flowExecutorRootConfig.flowConfig.quickFlowListener) {
                    flowExecutorRootConfig.flowConfig.quickFlowListener.beforeExecuteFlow(flow, flowExecutorRootConfig.flowContext);
                }
                long startTime = System.currentTimeMillis();
                try {
                    flow.executeBusinessFlow(flowExecutorRootConfig.flowContext);
                    long endTime = System.currentTimeMillis();
                    flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]"+(flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + flow.name() + flowExecutorRootConfig.remarkBuilder.toString());
                }catch (BrokenCurrentFlowException e){
                    long endTime = System.currentTimeMillis();
                    flowExecutorRootConfig.brokenReason = e.getReason();
                    flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]"+(flowExecutorRootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "")+"|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorRootConfig.remarkBuilder.toString());
                }finally {
                    flowExecutorRootConfig.remarkBuilder.setLength(0);
                }
                if (null != flowExecutorRootConfig.flowConfig.quickFlowListener) {
                    flowExecutorRootConfig.flowConfig.quickFlowListener.afterExecuteFlow(flow, flowExecutorRootConfig.flowContext);
                }
            }
        } catch (BrokenException e){
            flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]"+ currentFlow.name() + flowExecutorRootConfig.remarkBuilder.toString());
            flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]|[x]|业务流程终止|原因:"+e.getReason());
            flowExecutorRootConfig.brokenReason = e.getReason();
        } catch (Exception e) {
            flowExecutorRootConfig.printFlowTrace("[" + (flowExecutorRootConfig.index.getAndIncrement()) + "]|[x]|"+currentFlow.name()+"|业务流程发生异常|异常信息:" + e.getClass().getName() + "-" + e.getMessage());
            executeTryCatchFinallyHandler(flowExecutorRootConfig.tryCatchFinallyHandler, "catch", e);
            executeTryCatchFinallyHandler(flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "catch", e);
            flowExecutorRootConfig.exception = e;
            if(flowExecutorRootConfig.ignoreException){
                e.printStackTrace();
            }else{
                throw e;
            }
        } finally {
            flowExecutorRootConfig.decrementFlowLevel();
            flowExecutorRootConfig.printFlowTrace("["+flowExecutorRootConfig.name+"]");

            executeTryCatchFinallyHandler(flowExecutorRootConfig.tryCatchFinallyHandler, "finally", null);
            executeTryCatchFinallyHandler(flowExecutorRootConfig.flowConfig.tryCatchFinallyHandler, "finally", null);
            //移除临时数据
            Set<String> temporaryKeySet = flowExecutorRootConfig.temporaryKeySet;
            for(String temporaryKey:temporaryKeySet){
                flowExecutorRootConfig.dataMap.remove(temporaryKey);
            }
        }
        return flowExecutorRootConfig.flowContext;
    }

    /**保存流程实例*/
    public FlowExecutorOption save(){
        if(flowExecutorRootConfig.flowConfig.businessMap.containsKey(flowExecutorRootConfig.name)){
            throw new IllegalArgumentException("无法保存该流程!流程名称已经存在!流程名称:"+flowExecutorRootConfig.name);
        }
        BusinessFlow businessFlow = new BusinessFlow() {
            @Override
            public void executeBusinessFlow(FlowContext flowContext) throws Exception {
                QuickFlowExecutor quickFlowExecutor = flowExecutorRootConfig.quickFlow.startFlow(flowExecutorRootConfig.name)
                        .putData(flowExecutorRootConfig.dataMap);
                if(null!=flowExecutorRootConfig.tryCatchFinallyHandler){
                    quickFlowExecutor.tryCatchFinallyHandler(flowExecutorRootConfig.tryCatchFinallyHandler);
                }
                if(null!=flowExecutorRootConfig.beforeAfterFlowHandler){
                    quickFlowExecutor.beforeAfterFlowHandler(flowExecutorRootConfig.beforeAfterFlowHandler);
                }
                for (BusinessFlow flow : flowExecutorRootConfig.flowList) {
                    quickFlowExecutor.next(flow);
                }
                quickFlowExecutor.execute();
            }

            @Override
            public String name() {
                return flowExecutorRootConfig.name;
            }
        };
        flowExecutorRootConfig.flowConfig.businessMap.put(flowExecutorRootConfig.name, businessFlow);

        FlowExecutorOption flowExecutorOption = new FlowExecutorOption();
        flowExecutorOption.name = flowExecutorRootConfig.name;
        flowExecutorOption.requestData = flowExecutorRootConfig.dataMap;
        flowExecutorOption.flowNameList = getFlowNameList();
        return flowExecutorOption;
    }

    private List<String> getFlowNameList(){
        List<String> flowNameList = new ArrayList<>();
        if(null!=flowExecutorRootConfig.tryCatchFinallyHandler){
            flowNameList.add("handler-tryCatchFinally:"+flowExecutorRootConfig.tryCatchFinallyHandler.name());
        }
        if(null!=flowExecutorRootConfig.beforeAfterFlowHandler){
            flowNameList.add("handler-beforeAfter:"+flowExecutorRootConfig.beforeAfterFlowHandler.name());
        }
        for (BusinessFlow flow : flowExecutorRootConfig.flowList) {
            flowNameList.add(flow.name());
        }
        return flowNameList;
    }

    /**执行try-catch-finally事件*/
    private void executeTryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler, String operation, Exception e) throws Exception {
        if(null==tryCatchFinallyHandler){
            return;
        }
        switch (operation){
            case "try":{tryCatchFinallyHandler.handleTry(flowExecutorRootConfig.flowContext);}break;
            case "catch":{tryCatchFinallyHandler.handleException(flowExecutorRootConfig.flowContext, e);}break;
            case "finally":{tryCatchFinallyHandler.handleFinally(flowExecutorRootConfig.flowContext);}break;
            default:{
                throw new IllegalArgumentException("不支持调用该方法!方法名:"+operation);
            }
        }
    }

    private void executeBeforeAfterFlowHandler(BeforeAfterFlowHandler beforeAfterFlowHandler) throws Exception {
        if(null==beforeAfterFlowHandler){
            return;
        }
        List<BusinessFlow> beforeFlowList = beforeAfterFlowHandler.getBeforeFlowList();
        if(null!=beforeFlowList&&!beforeFlowList.isEmpty()){
            this.flowExecutorRootConfig.flowList.addAll(0, beforeFlowList);
        }
        List<BusinessFlow> afterFlowList = beforeAfterFlowHandler.getAfterFlowList();
        if(null!=afterFlowList&&!afterFlowList.isEmpty()){
            this.flowExecutorRootConfig.flowList.addAll(afterFlowList);
        }
    }
}
