package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Set;

public class QuickFlowExecutor {
    private Logger logger = LoggerFactory.getLogger(QuickFlowExecutor.class);

    /**配置信息*/
    private FlowExecutorConfig flowExecutorConfig;

    public QuickFlowExecutor(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(String key, Object value) {
        flowExecutorConfig.flowContext.putData(key, value);
        flowExecutorConfig.rootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putData(dataMap);
        flowExecutorConfig.rootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(String key, Object value) {
        flowExecutorConfig.flowContext.putTemporaryData(key, value);
        flowExecutorConfig.rootConfig.requestKeySet.add(key);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putTemporaryData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putTemporaryData(dataMap);
        flowExecutorConfig.rootConfig.requestKeySet.addAll(dataMap.keySet());
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(String key, Object value) {
        flowExecutorConfig.flowContext.putThreadLocalData(key, value);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putThreadLocalData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putThreadLocalData(dataMap);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(String flowName) {
        if(null==flowExecutorConfig.rootConfig.flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        BusinessFlow flow = flowExecutorConfig.rootConfig.flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName);
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(BusinessFlow flow) {
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlowExecutor tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowExecutorConfig.rootConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 是否打印路径
     */
    public QuickFlowExecutor printTrace(boolean printTrace) {
        flowExecutorConfig.rootConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略异常
     */
    public QuickFlowExecutor ignoreException(boolean ignoreException) {
        flowExecutorConfig.rootConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 执行业务逻辑
     */
    public FlowContext execute() throws Exception {
        printFlowTrace("|---"+flowExecutorConfig.name);
        addFlowName();
        BusinessFlow currentFlow = flowExecutorConfig.flowList.get(0);
        try {
            if(null!=flowExecutorConfig.rootConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);
            }
            //仅主流程才执行全局try事件
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);
            }
            //执行任务
            for (BusinessFlow flow : flowExecutorConfig.flowList) {
                currentFlow = flow;
                if (null != flowExecutorConfig.rootConfig.flowConfig.quickFlowListener) {
                    flowExecutorConfig.rootConfig.flowConfig.quickFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowContext);
                }
                long startTime = System.currentTimeMillis();
                try {
                    flow.executeBusinessFlow(flowExecutorConfig.flowContext);
                    long endTime = System.currentTimeMillis();
                    printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]"+(flowExecutorConfig.rootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + flow.name() + flowExecutorConfig.rootConfig.remarkBuilder.toString());
                }catch (BrokenCurrentFlowException e){
                    long endTime = System.currentTimeMillis();
                    printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]"+(flowExecutorConfig.rootConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + flow.name() + flowExecutorConfig.rootConfig.remarkBuilder.toString());
                    flowExecutorConfig.rootConfig.brokenReason = e.getReason();
                    printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.rootConfig.remarkBuilder.toString());
                }finally {
                    flowExecutorConfig.rootConfig.remarkBuilder.setLength(0);
                }
                if (null != flowExecutorConfig.rootConfig.flowConfig.quickFlowListener) {
                    flowExecutorConfig.rootConfig.flowConfig.quickFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowContext);
                }
            }
        } catch (BrokenException e){
            printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]"+ currentFlow.name() + flowExecutorConfig.rootConfig.remarkBuilder.toString());
            //非主流程继续抛出中断异常
            if(null!=flowExecutorConfig.parentFlowExecutorConfig&&!flowExecutorConfig.rootConfig.ignoreException){
                throw e;
            }
            printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]|[x]|业务流程终止|原因:"+e.getReason());
            flowExecutorConfig.rootConfig.brokenReason = e.getReason();
        } catch (Exception e) {
            printFlowTrace("[" + (flowExecutorConfig.rootConfig.index.getAndIncrement()) + "]|[x]|"+currentFlow.name()+"|业务流程发生异常|异常信息:" + e.getClass().getName() + "-" + e.getMessage());
            if(null!=flowExecutorConfig.rootConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);
            }
            //仅主流程才执行全局handleException事件
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);
            }
            //仅记录最上层流程的抛出异常
            if(null==flowExecutorConfig.parentFlowExecutorConfig){
                flowExecutorConfig.rootConfig.exception = e;
            }
            if(!flowExecutorConfig.rootConfig.ignoreException){
                throw e;
            }else{
                e.printStackTrace();
            }
        } finally {
            if(null!=flowExecutorConfig.rootConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
            }
            //仅主流程才执行全局handleFinally事件
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.rootConfig.flowConfig.tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
            }
            if(null==flowExecutorConfig.parentFlowExecutorConfig){
                //移除临时数据
                Set<String> temporaryKeySet = flowExecutorConfig.rootConfig.temporaryKeySet;
                for(String temporaryKey:temporaryKeySet){
                    flowExecutorConfig.rootConfig.dataMap.remove(temporaryKey);
                }
            }
        }
        return flowExecutorConfig.flowContext;
    }

    /**保存流程实例*/
    public QuickFlowExecutor save(){
        if(flowExecutorConfig.rootConfig.flowConfig.businessMap.containsKey(flowExecutorConfig.name)){
            throw new IllegalArgumentException("无法保存该流程!流程名称已经存在!流程名称:"+flowExecutorConfig.name);
        }
        BusinessFlow businessFlow = new BusinessFlow() {
            @Override
            public void executeBusinessFlow(FlowContext flowContext) throws Exception {
                flowContext.putData(flowExecutorConfig.rootConfig.dataMap);
                QuickFlowExecutor quickFlowExecutor = flowContext.startSubFlow(flowExecutorConfig.name);
                if(null!=flowExecutorConfig.rootConfig.tryCatchFinallyHandler){
                    quickFlowExecutor.tryCatchFinallyHandler(flowExecutorConfig.rootConfig.tryCatchFinallyHandler);
                }
                for (BusinessFlow flow : flowExecutorConfig.flowList) {
                    quickFlowExecutor.next(flow);
                }
                quickFlowExecutor.execute();
            }

            @Override
            public String name() {
                return flowExecutorConfig.name;
            }
        };
        flowExecutorConfig.rootConfig.flowConfig.businessMap.put(flowExecutorConfig.name, businessFlow);
        return this;
    }

    private void printFlowTrace(String content){
        if (flowExecutorConfig.rootConfig.printTrace&&flowExecutorConfig.rootConfig.flowConfig.printTrace) {
            for(int i=0;i<flowExecutorConfig.subflowLevel;i++){
                flowExecutorConfig.rootConfig.printTraceBuilder.append("\t");
            }
            flowExecutorConfig.rootConfig.printTraceBuilder.append(content + "\r\n");
        }
    }

    private void addFlowName(){
        if (flowExecutorConfig.rootConfig.printTrace&&flowExecutorConfig.rootConfig.flowConfig.printTrace) {
            if(null!=flowExecutorConfig.rootConfig.tryCatchFinallyHandler){
                this.flowExecutorConfig.rootConfig.flowNameList.add("handler:"+flowExecutorConfig.rootConfig.tryCatchFinallyHandler.name());
            }
            for (BusinessFlow flow : flowExecutorConfig.flowList) {
                this.flowExecutorConfig.rootConfig.flowNameList.add(flow.name());
            }
        }
    }

}
