package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.*;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.FlowNameInstanceMappingHandler;
import cn.schoolwow.quickflow.listener.QuickFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;

public class QuickFlow {
    private Logger logger = LoggerFactory.getLogger(QuickFlow.class);

    /**默认流程配置对象*/
    private static QuickFlow defaultQuickFlow = new QuickFlow();

    /**流程配置信息*/
    private FlowConfig flowConfig = new FlowConfig();

    public static QuickFlow newInstance() {
        return new QuickFlow();
    }

    /**获取默认QuickFlow对象*/
    public static QuickFlow defaultQuickFlow(){
        return defaultQuickFlow;
    }

    /**
     * 事件监听
     */
    public QuickFlow flowListener(QuickFlowListener quickFlowListener) {
        flowConfig.quickFlowListener = quickFlowListener;
        return this;
    }

    /**
     * 是否打印执行路径
     */
    public QuickFlow printTrace(boolean printTrace) {
        flowConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否记录流程耗时
     */
    public QuickFlow printConsumeTime(boolean printConsumeTime) {
        flowConfig.printConsumeTime = printConsumeTime;
        return this;
    }

    /**
     * 指定流程名称和实例映射
     */
    public QuickFlow flowNameInstanceMappingHandler(FlowNameInstanceMappingHandler flowNameInstanceMappingHandler) {
        flowConfig.flowNameInstanceMappingHandler = flowNameInstanceMappingHandler;
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlow tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(String name) {
        if(null==name||name.isEmpty()){
            throw new IllegalArgumentException("请指定主流程名称!");
        }
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        {
            FlowExecutorRootConfig rootConfig = new FlowExecutorRootConfig();
            rootConfig.name = name;
            rootConfig.threadLocalDataMap.set(new HashMap<>());
            rootConfig.flowConfig = flowConfig;
            if(flowConfig.printTrace){
                rootConfig.printTraceBuilder = new StringBuilder();
            }
            flowExecutorConfig.rootConfig = rootConfig;
        }
        flowExecutorConfig.name = name;
        flowExecutorConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        flowExecutorConfig.subflowLevel = 0;
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    /**执行流程*/
    public QuickFlowExecutor startFlow(FlowExecutorOption flowExecutorOption) {
        if(null==flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        if(null==flowExecutorOption.flowNameList||flowExecutorOption.flowNameList.isEmpty()){
            throw new IllegalArgumentException("请指定流程列表!");
        }
        QuickFlowExecutor quickFlowExecutor = startFlow(flowExecutorOption.name)
                .putData(flowExecutorOption.requestData);
        for(String flowName:flowExecutorOption.flowNameList){
            if(flowName.startsWith("handler:")){
                TryCatchFinallyHandler tryCatchFinallyHandler = flowConfig.flowNameInstanceMappingHandler.getTryCatchFinallyByName(flowName.substring(8));
                quickFlowExecutor.tryCatchFinallyHandler(tryCatchFinallyHandler);
            }else if(flowConfig.businessMap.containsKey(flowName)){
                quickFlowExecutor.next(flowConfig.businessMap.get(flowName));
            }else{
                quickFlowExecutor.next(flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName));
            }
        }
        return quickFlowExecutor;
    }

    /**获取流程实例*/
    public BusinessFlow getBusinessFlow(String flowName) {
        BusinessFlow businessFlow = flowConfig.businessMap.get(flowName);
        if(null==businessFlow){
            throw new IllegalArgumentException("流程实例不存在!请先调用save方法保存该流程!流程名称:"+flowName);
        }
        return businessFlow;
    }
}
