package cn.schoolwow.quickflow.domain;

import cn.schoolwow.quickflow.QuickFlowExecutor;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.util.*;

public class FlowContextImpl implements FlowContext {
    /**
     * 配置数据
     */
    private FlowExecutorConfig flowExecutorConfig;

    public FlowContextImpl(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    @Override
    public Object checkData(String key) {
        Object value = getData(key);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public <T> T checkData(String key, Class<T> clazz) {
        T value = getData(key, clazz);
        if(null==value){
            throw new IllegalArgumentException("上下文数据中key为"+key+"的值不能为空!");
        }
        if(value instanceof Collection){
            if(((Collection)value).isEmpty()){
                throw new IllegalArgumentException("上下文数据中key为"+key+"的列表不能为空!");
            }
        }
        return value;
    }

    @Override
    public Object getData(String key) {
        Object value = flowExecutorConfig.dataMap.get(key);
        if(null==value){
            value = flowExecutorConfig.threadLocalDataMap.get().get(key);
        }
        return value;
    }

    @Override
    public <T> T getData(String key, Class<T> clazz) {
        Object value = getData(key);
        if(value instanceof JSONObject){
            JSONObject o = (JSONObject) value;
            return o.toJavaObject(clazz);
        }
        if(value instanceof JSONArray){
            JSONArray o = (JSONArray) value;
            return o.toJavaObject(clazz);
        }
        try {
            T t = clazz.cast(value);
            return t;
        }catch (Exception e){
            throw new UnsupportedOperationException("无法转换类型!预期类型:"+clazz.getName()+",实际类型:"+value.getClass().getName());
        }
    }

    @Override
    public Object getData(String key, Object defaultValue) {
        Object value = getData(key);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public <T> T getData(String key, Class<T> clazz, T defaultValue) {
        T value = getData(key, clazz);
        if(null==value){
            return defaultValue;
        }else{
            return value;
        }
    }

    @Override
    public Map<String, Object> getData() {
        return flowExecutorConfig.dataMap;
    }

    @Override
    public void putData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorConfig.dataMap.put(key, value);
        }
    }

    @Override
    public void putData(Map<String, Object> map) {
        if(null!=map){
            Iterator<Map.Entry<String,Object>> iterator = map.entrySet().iterator();
            while(iterator.hasNext()){
                if(null==iterator.next().getValue()){
                    iterator.remove();
                }
            }
            flowExecutorConfig.dataMap.putAll(map);
        }
    }

    @Override
    public Map<String, Object> getTemporaryDataMap() {
        Map<String,Object> temporaryDataMap = new HashMap<>();
        for(String key:flowExecutorConfig.temporaryKeySet){
            temporaryDataMap.put(key, flowExecutorConfig.dataMap.get(key));
        }
        return temporaryDataMap;
    }

    @Override
    public Map<String, Object> getRequestDataMap() {
        Map<String,Object> requestDataMap = new HashMap<>();
        for(String key:flowExecutorConfig.requestKeySet){
            requestDataMap.put(key, flowExecutorConfig.dataMap.get(key));
        }
        return requestDataMap;
    }

    @Override
    public Map<String, Object> getContextDataMap() {
        Set<String> keySet = new HashSet<>(flowExecutorConfig.dataMap.keySet());
        keySet.removeAll(flowExecutorConfig.requestKeySet);
        keySet.removeAll(flowExecutorConfig.temporaryKeySet);
        Map<String,Object> contextDataMap = new HashMap<>();
        for(String key:keySet){
            contextDataMap.put(key, flowExecutorConfig.dataMap.get(key));
        }
        return contextDataMap;
    }

    @Override
    public void putTemporaryData(String key, Object value) {
        putData(key, value);
        if(null!=key){
            flowExecutorConfig.temporaryKeySet.add(key);
        }
    }

    @Override
    public void putTemporaryData(Map<String, Object> map) {
        putData(map);
        if(null!=map){
            flowExecutorConfig.temporaryKeySet.addAll(map.keySet());
        }
    }

    @Override
    public void putThreadLocalData(String key, Object value) {
        if(null!=key&&null!=value){
            flowExecutorConfig.threadLocalDataMap.get().put(key, value);
        }
    }

    @Override
    public void putThreadLocalData(Map<String, Object> map) {
        if(null!=map){
            flowExecutorConfig.threadLocalDataMap.get().putAll(map);
        }
    }

    @Override
    public void removeData(String key) {
        if(null!=key){
            flowExecutorConfig.dataMap.remove(key);
        }
    }

    @Override
    public void remark(String remark) {
        flowExecutorConfig.remarkBuilder.append("|"+remark);
    }

    @Override
    public void brokenCurrentFlow(String reason) {
        throw new BrokenCurrentFlowException(reason);
    }

    @Override
    public void broken(String reason) {
        throw new BrokenException(reason);
    }

    @Override
    public String getFlowName() {
        return flowExecutorConfig.name;
    }

    @Override
    public List<String> getFlowNameList() {
        return flowExecutorConfig.flowNameList;
    }

    @Override
    public String getFlowTrace() {
        return flowExecutorConfig.printTraceBuilder.toString();
    }

    @Override
    public Exception getFlowException() {
        return flowExecutorConfig.exception;
    }

    @Override
    public String getBrokenReason() {
        return flowExecutorConfig.brokenReason;
    }

    @Override
    public QuickFlowExecutor startSubFlow(String name) {
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;
        flowExecutorConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        flowExecutorConfig.dataMap = this.flowExecutorConfig.dataMap;
        flowExecutorConfig.threadLocalDataMap.set(new HashMap<>());
        flowExecutorConfig.threadLocalDataMap.get().putAll(this.flowExecutorConfig.dataMap);
        flowExecutorConfig.index = this.flowExecutorConfig.index;
        flowExecutorConfig.subflowLevel = this.flowExecutorConfig.subflowLevel+1;
        flowExecutorConfig.flowNameList = new ArrayList<>();
        flowExecutorConfig.temporaryKeySet = this.flowExecutorConfig.temporaryKeySet;
        flowExecutorConfig.parentFlowExecutorConfig = this.flowExecutorConfig;
        flowExecutorConfig.flowConfig = this.flowExecutorConfig.flowConfig;
        flowExecutorConfig.printTrace = this.flowExecutorConfig.printTrace;
        if(flowExecutorConfig.flowConfig.printTrace){
            flowExecutorConfig.printTraceBuilder = this.flowExecutorConfig.printTraceBuilder;
        }
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    @Override
    public BusinessFlow getBusinessFlow(String flowName) {
        BusinessFlow businessFlow = flowExecutorConfig.flowConfig.businessMap.get(flowName);
        if(null==businessFlow){
            throw new IllegalArgumentException("流程实例不存在!请先调用save方法保存该流程!流程名称:"+flowName);
        }
        return businessFlow;
    }

}
