package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.exception.BrokenCurrentFlowException;
import cn.schoolwow.quickflow.exception.BrokenException;
import cn.schoolwow.quickflow.flow.AsyncBusinessFlow;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.flow.ConditionFlow;
import cn.schoolwow.quickflow.flow.Flow;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Set;

public class QuickFlowExecutor {
    private Logger logger = LoggerFactory.getLogger(QuickFlowExecutor.class);

    /**配置信息*/
    private FlowExecutorConfig flowExecutorConfig;

    public QuickFlowExecutor(FlowExecutorConfig flowExecutorConfig) {
        this.flowExecutorConfig = flowExecutorConfig;
    }

    /**
     * 获取上下文参数
     */
    public Object checkData(String key) {
        return flowExecutorConfig.flowContext.checkData(key);
    }

    /**
     * 获取上下文参数
     */
    public Object getData(String key) {
        return flowExecutorConfig.flowContext.getData(key);
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(String key, Object value) {
        flowExecutorConfig.flowContext.putData(key, value);
        return this;
    }

    /**
     * 设置上下文参数
     */
    public QuickFlowExecutor putData(Map<String,Object> dataMap) {
        flowExecutorConfig.flowContext.putData(dataMap);
        return this;
    }

    /**
     * 获取上下文对象
     */
    public FlowContext getFlowContext() {
        return flowExecutorConfig.flowContext;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(String flowName) {
        if(null==flowExecutorConfig.flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        Flow flow = flowExecutorConfig.flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName);
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定下一个业务逻辑
     */
    public QuickFlowExecutor next(Flow flow) {
        flowExecutorConfig.flowList.add(flow);
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlowExecutor tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowExecutorConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 是否打印路径
     */
    public QuickFlowExecutor printTrace(boolean printTrace) {
        flowExecutorConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否忽略异常
     */
    public QuickFlowExecutor ignoreException(boolean ignoreException) {
        flowExecutorConfig.ignoreException = ignoreException;
        return this;
    }

    /**
     * 执行业务逻辑
     */
    public QuickFlowExecutor execute() throws Exception {
        printFlowTrace("|---"+flowExecutorConfig.name);
        if(null==flowExecutorConfig.parentFlowExecutorConfig){
            if(null!=flowExecutorConfig.tryCatchFinallyHandler){
                this.flowExecutorConfig.flowNameList.add("handler:"+flowExecutorConfig.tryCatchFinallyHandler.name());
            }
            //只记录主流程
            for (Flow flow : flowExecutorConfig.flowList) {
                this.flowExecutorConfig.flowNameList.add(flow.name());
            }
        }
        Flow currentFlow = flowExecutorConfig.flowList.get(0);
        try {
            if(null!=flowExecutorConfig.tryCatchFinallyHandler){
                flowExecutorConfig.tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);
            }
            //仅主流程才执行全局try事件
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.flowConfig.tryCatchFinallyHandler.handleTry(flowExecutorConfig.flowContext);
            }
            //执行任务
            for (Flow flow : flowExecutorConfig.flowList) {
                currentFlow = flow;
                if (null != flowExecutorConfig.flowConfig.quickFlowListener) {
                    flowExecutorConfig.flowConfig.quickFlowListener.beforeExecuteFlow(flow, flowExecutorConfig.flowContext);
                }
                try {
                    executeFlow(flow);
                }catch (BrokenCurrentFlowException e){
                    flowExecutorConfig.brokenReason = e.getReason();
                    printFlowTrace("[" + (flowExecutorConfig.index.getAndIncrement()) + "]|[x]|"+ flow.name() + "|当前流程中断|中断原因:" + e.getReason() + flowExecutorConfig.remarkBuilder.toString());
                    flowExecutorConfig.remarkBuilder.setLength(0);
                }
                if (null != flowExecutorConfig.flowConfig.quickFlowListener) {
                    flowExecutorConfig.flowConfig.quickFlowListener.afterExecuteFlow(flow, flowExecutorConfig.flowContext);
                }
            }
        } catch (BrokenException e){
            //非主流程继续抛出中断异常
            if(null!=flowExecutorConfig.parentFlowExecutorConfig&&!flowExecutorConfig.ignoreException){
                throw e;
            }
            printFlowTrace("[" + (flowExecutorConfig.index.getAndIncrement()) + "]|[x]|业务流程终止|原因:"+e.getReason());
            flowExecutorConfig.brokenReason = e.getReason();
        } catch (Exception e) {
            printFlowTrace("[" + (flowExecutorConfig.index.getAndIncrement()) + "]|[x]|"+currentFlow.name()+"|业务流程发生异常|异常信息:" + e.getClass().getName() + "-" + e.getMessage());
            if(null!=flowExecutorConfig.tryCatchFinallyHandler){
                flowExecutorConfig.tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);
            }
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.flowConfig.tryCatchFinallyHandler.handleException(flowExecutorConfig.flowContext, e);
            }
            if(null==flowExecutorConfig.parentFlowExecutorConfig){
                flowExecutorConfig.exception = e;
            }
            if(!flowExecutorConfig.ignoreException){
                throw e;
            }else{
                e.printStackTrace();
            }
        } finally {
            if(null!=flowExecutorConfig.tryCatchFinallyHandler){
                flowExecutorConfig.tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
            }
            if(null==flowExecutorConfig.parentFlowExecutorConfig&&null!=flowExecutorConfig.flowConfig.tryCatchFinallyHandler){
                flowExecutorConfig.flowConfig.tryCatchFinallyHandler.handleFinally(flowExecutorConfig.flowContext);
            }
            if(null==flowExecutorConfig.parentFlowExecutorConfig){
                //移除临时数据
                Set<String> temporaryKeySet = flowExecutorConfig.temporaryKeySet;
                for(String temporaryKey:temporaryKeySet){
                    flowExecutorConfig.dataMap.remove(temporaryKey);
                }
            }
        }
        return this;
    }

    /**保存流程实例*/
    public QuickFlowExecutor save(){
        if(flowExecutorConfig.flowConfig.businessMap.containsKey(flowExecutorConfig.name)){
            throw new IllegalArgumentException("无法保存该流程!流程名称已经存在!流程名称:"+flowExecutorConfig.name);
        }
        BusinessFlow businessFlow = new BusinessFlow() {
            @Override
            public void executeBusinessFlow(FlowContext flowContext) throws Exception {
                QuickFlowExecutor quickFlowExecutor = flowContext.startSubFlow(flowExecutorConfig.name);
                if(null!=flowExecutorConfig.tryCatchFinallyHandler){
                    quickFlowExecutor.tryCatchFinallyHandler(flowExecutorConfig.tryCatchFinallyHandler);
                }
                for (Flow flow : flowExecutorConfig.flowList) {
                    quickFlowExecutor.next(flow);
                }
                quickFlowExecutor.execute();
            }

            @Override
            public String name() {
                return flowExecutorConfig.name;
            }
        };
        flowExecutorConfig.flowConfig.businessMap.put(flowExecutorConfig.name, businessFlow);
        return this;
    }

    /**
     * 执行业务逻辑
     */
    private void executeFlow(Flow flow) throws Exception {
        long startTime = System.currentTimeMillis();
        if (flow instanceof BusinessFlow) {
            ((BusinessFlow) flow).executeBusinessFlow(flowExecutorConfig.flowContext);
        } else if (flow instanceof AsyncBusinessFlow) {
            flowExecutorConfig.flowConfig.threadPoolExecutor.execute(() -> {
                try {
                    ((AsyncBusinessFlow) flow).asyncExecuteBusinessFlow(flowExecutorConfig.flowContext);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            });
        } else if (flow instanceof ConditionFlow) {
            Flow conditionFlow = ((ConditionFlow) flow).judgeBusinessFlow(flowExecutorConfig.flowContext);
            if (conditionFlow.getClass().getName().equals(flow.getClass().getName())) {
                throw new IllegalArgumentException("发现循环业务逻辑!循环业务类:" + conditionFlow.getClass().getName());
            }
            executeFlow(conditionFlow);
        } else {
            throw new UnsupportedOperationException("不支持的业务类型!" + flow.getClass().getName());
        }
        long endTime = System.currentTimeMillis();
        printFlowTrace("[" + (flowExecutorConfig.index.getAndIncrement()) + "]"+(flowExecutorConfig.flowConfig.printConsumeTime ? ("|" + (endTime - startTime) + "毫秒|") : "") + flow.name() + flowExecutorConfig.remarkBuilder.toString());
        flowExecutorConfig.remarkBuilder.setLength(0);
    }

    private void printFlowTrace(String content){
        if (flowExecutorConfig.printTrace&&flowExecutorConfig.flowConfig.printTrace) {
            for(int i=0;i<flowExecutorConfig.subflowLevel;i++){
                flowExecutorConfig.printTraceBuilder.append("\t");
            }
            flowExecutorConfig.printTraceBuilder.append(content + "\r\n");
        }
    }
}
