package cn.schoolwow.quickflow;

import cn.schoolwow.quickflow.domain.FlowConfig;
import cn.schoolwow.quickflow.domain.FlowContextImpl;
import cn.schoolwow.quickflow.domain.FlowExecutorConfig;
import cn.schoolwow.quickflow.domain.FlowExecutorOption;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import cn.schoolwow.quickflow.listener.FlowNameInstanceMappingHandler;
import cn.schoolwow.quickflow.listener.QuickFlowListener;
import cn.schoolwow.quickflow.listener.TryCatchFinallyHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;

public class QuickFlow {
    private Logger logger = LoggerFactory.getLogger(QuickFlow.class);

    /**流程配置信息*/
    private FlowConfig flowConfig = new FlowConfig();

    private boolean build;

    public static QuickFlow newInstance() {
        return new QuickFlow();
    }

    /**
     * 指定异步任务执行线程池
     */
    public QuickFlow threadPool(ThreadPoolExecutor threadPoolExecutor) {
        flowConfig.threadPoolExecutor = threadPoolExecutor;
        return this;
    }

    /**
     * 事件监听
     */
    public QuickFlow flowListener(QuickFlowListener quickFlowListener) {
        flowConfig.quickFlowListener = quickFlowListener;
        return this;
    }

    /**
     * 是否打印执行路径
     */
    public QuickFlow printTrace(boolean printTrace) {
        flowConfig.printTrace = printTrace;
        return this;
    }

    /**
     * 是否记录流程耗时
     */
    public QuickFlow printConsumeTime(boolean printConsumeTime) {
        flowConfig.printConsumeTime = printConsumeTime;
        return this;
    }

    /**
     * 指定流程名称和实例映射
     */
    public QuickFlow flowNameInstanceMappingHandler(FlowNameInstanceMappingHandler flowNameInstanceMappingHandler) {
        flowConfig.flowNameInstanceMappingHandler = flowNameInstanceMappingHandler;
        return this;
    }

    /**
     * 指定try-catch-finally处理器
     */
    public QuickFlow tryCatchFinallyHandler(TryCatchFinallyHandler tryCatchFinallyHandler) {
        flowConfig.tryCatchFinallyHandler = tryCatchFinallyHandler;
        return this;
    }

    /**
     * 生成对象
     */
    public QuickFlow build() {
        //判断有无异步任务
        if (null == flowConfig.threadPoolExecutor) {
            flowConfig.threadPoolExecutor = (ThreadPoolExecutor) Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors() * 2);
        }
        build = true;
        return this;
    }

    /**
     * 构建流程
     * */
    public QuickFlowExecutor startFlow(String name) {
        if(null==name||name.isEmpty()){
            throw new IllegalArgumentException("请指定主流程名称!");
        }
        if (!build) {
            throw new UnsupportedOperationException("请先调用build方法!");
        }
        FlowExecutorConfig flowExecutorConfig = new FlowExecutorConfig();
        flowExecutorConfig.name = name;
        flowExecutorConfig.flowContext = new FlowContextImpl(flowExecutorConfig);
        flowExecutorConfig.dataMap = new ConcurrentHashMap<>();
        flowExecutorConfig.flowConfig = flowConfig;
        if(flowConfig.printTrace){
            flowExecutorConfig.printTraceBuilder = new StringBuilder();
        }
        flowExecutorConfig.flowNameList = new ArrayList<>();
        flowExecutorConfig.temporaryKeySet = new HashSet<>();
        return new QuickFlowExecutor(flowExecutorConfig);
    }

    /**执行流程*/
    public QuickFlowExecutor startFlow(FlowExecutorOption flowExecutorOption) {
        if(null==flowConfig.flowNameInstanceMappingHandler){
            throw new IllegalArgumentException("请先指定流程名称实例映射处理器!");
        }
        if(null==flowExecutorOption.flowNameList||flowExecutorOption.flowNameList.isEmpty()){
            throw new IllegalArgumentException("请指定流程列表!");
        }
        QuickFlowExecutor quickFlowExecutor = startFlow(flowExecutorOption.name)
                .putData(flowExecutorOption.dataMap);
        for(String flowName:flowExecutorOption.flowNameList){
            if(flowName.startsWith("handler:")){
                TryCatchFinallyHandler tryCatchFinallyHandler = flowConfig.flowNameInstanceMappingHandler.getTryCatchFinallyByName(flowName.substring(8));
                quickFlowExecutor.tryCatchFinallyHandler(tryCatchFinallyHandler);
            }else if(flowConfig.businessMap.containsKey(flowName)){
                quickFlowExecutor.next(flowConfig.businessMap.get(flowName));
            }else{
                quickFlowExecutor.next(flowConfig.flowNameInstanceMappingHandler.getFlowByName(flowName));
            }
        }
        return quickFlowExecutor;
    }

    /**获取流程实例*/
    public BusinessFlow getBusinessFlow(String flowName) {
        BusinessFlow businessFlow = flowConfig.businessMap.get(flowName);
        if(null==businessFlow){
            throw new IllegalArgumentException("流程实例不存在!请先调用save方法保存该流程!流程名称:"+flowName);
        }
        return businessFlow;
    }
}
