package cn.schoolwow.quickdao.flow.ddl.property;

import cn.schoolwow.quickdao.annotation.IdStrategy;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.DatabaseType;
import cn.schoolwow.quickdao.domain.internal.common.ResultSetConsumer;
import cn.schoolwow.quickdao.flow.executor.ExecuteQueryConnectionFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;

import java.sql.ResultSet;
import java.util.Arrays;

public class GetPropertyFlow implements BusinessFlow {
    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        DatabaseType databaseType = (DatabaseType) flowContext.checkData("databaseType");
        switch (databaseType){
            case H2:{
                getByH2(flowContext);
            }break;
            case SQLite:{
                getBySQLite(flowContext);
            }break;
            case Mysql:
            case MariaDB:{
                getByMysql(flowContext);
            }break;
            case Postgresql:{
                getByPostgres(flowContext);
            }break;
            case SQLServer:{
                getBySQLServer(flowContext);
            }break;
            case Oracle:{
                getByOracle(flowContext);
            }break;
        }
        Property property = (Property) flowContext.getData("property");
        if(null!=property){
            if (property.columnType.contains("(") && property.columnType.contains(")")) {
                property.range = property.columnType.substring(property.columnType.indexOf("(") + 1, property.columnType.indexOf(")"));
                property.columnType = property.columnType.substring(0, property.columnType.indexOf("("));
            }
        }
    }

    @Override
    public String name() {
        return "获取数据库表字段列表";
    }

    private void getByH2(FlowContext flowContext) {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表指定字段信息")
                .putTemporaryData("sql", "select type_name, is_nullable, column_default from information_schema.`columns` where table_schema = '"+quickDAOConfig.databaseContext.databaseName+"' and table_name = ? and column_name = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            Property property = new Property();
                            property.column = resultSet.getString("column_name");
                            //无符号填充0 => float unsigned zerofill
                            property.columnType = resultSet.getString("type_name");
                            if (property.columnType.contains(" ")) {
                                property.columnType = property.columnType.substring(0, property.columnType.indexOf(" "));
                            }
                            property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                            if (null != resultSet.getString("column_default")) {
                                property.defaultValue = resultSet.getString("column_default");
                            }
                            flowContext.putData("property", property);
                        }
                    }
                })
                .execute();
    }

    private void getBySQLite(FlowContext flowContext) {
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表指定字段信息")
                .putTemporaryData("sql", "PRAGMA table_info(`" + tableName + "`);")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while (resultSet.next()) {
                            String resultSetColumnName = resultSet.getString("name");
                            if(resultSetColumnName.equalsIgnoreCase(columnName)){
                                Property property = new Property();
                                property.column = columnName;
                                property.columnType = resultSet.getString("type");
                                property.notNull = "1".equals(resultSet.getString("notnull"));
                                if (null != resultSet.getString("dflt_value")) {
                                    property.defaultValue = resultSet.getString("dflt_value");
                                }
                                if (1 == resultSet.getInt("pk")) {
                                    property.id = true;
                                    property.strategy = IdStrategy.AutoIncrement;
                                }
                                flowContext.putData("property", property);
                                break;
                            }
                        }
                    }
                })
                .execute();
    }

    private void getByMysql(FlowContext flowContext) {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段信息")
                .putTemporaryData("sql", "select column_name, column_type, is_nullable, column_key, extra, column_default, column_comment from information_schema.`columns` where table_schema = '"+quickDAOConfig.databaseContext.databaseName+"' and table_name = ? and column_name = ? order by table_name asc, ordinal_position asc;")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            Property property = new Property();
                            property.column = resultSet.getString("column_name");
                            //无符号填充0 => float unsigned zerofill
                            property.columnType = resultSet.getString("column_type");
                            if (property.columnType.contains(" ")) {
                                property.columnType = property.columnType.substring(0, property.columnType.indexOf(" ")).trim();
                            }
                            property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                            String key = resultSet.getString("column_key");
                            if ("PRI".equals(key)) {
                                property.id = true;
                            }
                            if ("auto_increment".equals(resultSet.getString("extra"))) {
                                property.id = true;
                                property.strategy = IdStrategy.AutoIncrement;
                            } else {
                                property.strategy = IdStrategy.None;
                            }
                            if (null != resultSet.getString("column_default")) {
                                property.defaultValue = resultSet.getString("column_default");
                                if (!property.defaultValue.contains("CURRENT_TIMESTAMP") && !property.defaultValue.contains("'")) {
                                    property.defaultValue = "'" + property.defaultValue + "'";
                                }
                            }
                            property.comment = resultSet.getString("column_comment").replace("\"", "\\\"");
                            flowContext.putData("property", property);
                        }
                    }
                })
                .execute();
    }

    private void getByPostgres(FlowContext flowContext) {
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段信息")
                .putTemporaryData("sql", "select attname as column_name, attnum as oridinal_position, attnotnull as notnull, format_type(atttypid,atttypmod) as type, col_description(attrelid, attnum) as comment from pg_attribute join pg_class on pg_attribute.attrelid = pg_class.oid where attnum > 0 and atttypid > 0 and pg_class.relname = ? and attname = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            Property property = new Property();
                            property.column = resultSet.getString("column_name");
                            property.columnType = resultSet.getString("type");
                            property.notNull = "t".equals(resultSet.getString("notnull"));
                            property.comment = resultSet.getString("comment");
                            flowContext.putData("property", property);
                        }
                    }
                })
                .execute();

        if(!flowContext.containKey("property")){
            flowContext.broken("字段不存在");
        }
        Property property = (Property) flowContext.checkData("property");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段类型信息")
                .putTemporaryData("sql", "select ordinal_position,column_default,is_nullable,udt_name,column_default from information_schema.columns where table_name = ? and column_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            property.columnType = resultSet.getString("udt_name");
                            if (null != resultSet.getString("column_default")) {
                                property.defaultValue = resultSet.getString("column_default");
                                if(property.defaultValue.startsWith("nextval(")){
                                    property.id = true;
                                    property.strategy = IdStrategy.AutoIncrement;
                                }
                            }
                        }
                    }
                })
                .execute();
    }

    private void getBySQLServer(FlowContext flowContext){
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段类型信息")
                .putTemporaryData("sql", "select ordinal_position,column_type,is_nullable from information_schema.columns where table_name = ? and column_name = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            Property property = new Property();
                            property.column = resultSet.getString("column_name");
                            property.columnType = resultSet.getString("column_type");
                            property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                            flowContext.putData("property", property);
                        }
                    }
                })
                .execute();

        if(flowContext.containKey("property")){
            flowContext.broken("字段不存在");
        }
        Property property = (Property) flowContext.checkData("property");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段注释")
                .putTemporaryData("sql", "select convert(varchar(255),a.value) value from sys.extended_properties a, sysobjects b, sys.columns c where a.major_id = b.id and c.object_id = b.id and c.column_id = a.minor_id and b.name = ? and c.name = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            property.comment = resultSet.getString("value");
                        }
                    }
                })
                .execute();

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表主键")
                .putTemporaryData("sql", "select column_name from information_schema.key_column_usage where table_name = ? and column_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            property.id = true;
                            property.strategy = IdStrategy.AutoIncrement;
                        }
                    }
                })
                .execute();
    }

    private void getByOracle(FlowContext flowContext) {
        String tableName = (String) flowContext.checkData("tableName");
        String columnName = (String) flowContext.checkData("columnName");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段信息")
                .putTemporaryData("sql", "select column_type, nullable, data_length from user_tab_columns where table_name = ? and column_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            Property property = new Property();
                            property.column = resultSet.getString("column_name");
                            property.columnType = resultSet.getString("column_type");
                            if(property.columnType.contains(" ")){
                                property.columnType = property.columnType.substring(0,property.columnType.indexOf(" "));
                            }
                            String dataLength = resultSet.getString("data_length");
                            if(property.columnType.toLowerCase().contains("char")&&null!=dataLength&&!dataLength.isEmpty()){
                                property.columnType += "(" + dataLength + ")";
                            }
                            property.notNull = "N".equals(resultSet.getString("nullable"));
                            flowContext.putData("property", property);
                        }
                    }
                })
                .execute();

        if(flowContext.containKey("property")){
            flowContext.broken("字段不存在");
        }
        Property property = (Property) flowContext.checkData("property");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取指定表字段注释")
                .putTemporaryData("sql", "select comments from user_col_comments where table_name = ? and column_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, columnName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            property.comment = resultSet.getString("comments");
                        }
                    }
                })
                .execute();
    }

}
