package cn.schoolwow.quickdao.flow.ddl.property;

import cn.schoolwow.quickdao.annotation.IdStrategy;
import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.DatabaseType;
import cn.schoolwow.quickdao.domain.internal.common.ResultSetConsumer;
import cn.schoolwow.quickdao.flow.executor.ExecuteQueryConnectionFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;

import java.sql.ResultSet;
import java.util.List;

public class GetEntityPropertyListFlow implements BusinessFlow {
    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        DatabaseType databaseType = (DatabaseType) flowContext.checkData("databaseType");

        flowContext.putTemporaryData("name", "获取数据库表字段列表");
        switch (databaseType){
            case H2:{
                getByH2(flowContext);
            }break;
            case SQLite:{
                getBySQLite(flowContext);
            }break;
            case Mysql:
            case MariaDB:{
                getByMysql(flowContext);
            }break;
            case Postgresql:{
                getByPostgres(flowContext);
            }break;
            case SQLServer:{
                getBySQLServer(flowContext);
            }break;
            case Oracle:{
                getByOracle(flowContext);
            }break;
        }
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");
        for(Entity entity:entityList){
            for(Property property: entity.properties){
                if (property.columnType.contains("(") && property.columnType.contains(")")) {
                    property.range = property.columnType.substring(property.columnType.indexOf("(") + 1, property.columnType.indexOf(")"));
                    property.columnType = property.columnType.substring(0, property.columnType.indexOf("("));
                }
            }
        }
    }

    @Override
    public String name() {
        return "获取数据库所有表字段列表";
    }

    private void getByH2(FlowContext flowContext) {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段信息")
                .putTemporaryData("sql", "select table_name, column_name, type_name, character_maximum_length, is_nullable, column_default from information_schema.`columns` where table_schema = '"+quickDAOConfig.databaseContext.databaseName+"';")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)){
                                    continue;
                                }
                                Property property = new Property();
                                property.column = resultSet.getString("column_name");
                                //无符号填充0 => float unsigned zerofill
                                property.columnType = resultSet.getString("type_name");
                                if (property.columnType.contains(" ")) {
                                    property.columnType = property.columnType.substring(0, property.columnType.indexOf(" "));
                                }
                                property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                                if (null != resultSet.getString("column_default")) {
                                    property.defaultValue = resultSet.getString("column_default");
                                }
                                entity.properties.add(property);
                            }
                        }
                    }
                })
                .execute();
    }

    private void getBySQLite(FlowContext flowContext) {
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");
        for(Entity entity:entityList){
            //获取表字段列表
            flowContext.startFlow(new ExecuteQueryConnectionFlow())
                    .putTemporaryData("name", "获取表字段信息")
                    .putTemporaryData("sql", "PRAGMA table_info(`" + entity.tableName + "`);")
                    .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                        @Override
                        public void consumeResultSet(ResultSet resultSet) throws Exception {
                            while (resultSet.next()) {
                                Property property = new Property();
                                property.column = resultSet.getString("name");
                                property.columnType = resultSet.getString("type");
                                property.notNull = "1".equals(resultSet.getString("notnull"));
                                if (null != resultSet.getString("dflt_value")) {
                                    property.defaultValue = resultSet.getString("dflt_value");
                                }
                                if (1 == resultSet.getInt("pk")) {
                                    property.id = true;
                                    property.strategy = IdStrategy.AutoIncrement;
                                }
                                entity.properties.add(property);
                            }
                        }
                    })
                    .execute();
        }
    }

    private void getByMysql(FlowContext flowContext) {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段信息")
                .putTemporaryData("sql", "select table_name, column_name, column_type, character_maximum_length, numeric_precision, is_nullable, column_key, extra, column_default, column_comment from information_schema.`columns` where table_schema = '"+quickDAOConfig.databaseContext.databaseName+"' order by table_name asc, ordinal_position asc;")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)){
                                    continue;
                                }
                                Property property = new Property();
                                property.column = resultSet.getString("column_name");
                                //无符号填充0 => float unsigned zerofill
                                property.columnType = resultSet.getString("column_type");
                                if (property.columnType.contains(" ")) {
                                    property.columnType = property.columnType.substring(0, property.columnType.indexOf(" ")).trim();
                                }
                                property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                                String key = resultSet.getString("column_key");
                                if ("PRI".equals(key)) {
                                    property.id = true;
                                }
                                if ("auto_increment".equals(resultSet.getString("extra"))) {
                                    property.id = true;
                                    property.strategy = IdStrategy.AutoIncrement;
                                } else {
                                    property.strategy = IdStrategy.None;
                                }
                                if (null != resultSet.getString("column_default")) {
                                    property.defaultValue = resultSet.getString("column_default");
                                    if (!property.defaultValue.contains("CURRENT_TIMESTAMP") && !property.defaultValue.contains("'")) {
                                        property.defaultValue = "'" + property.defaultValue + "'";
                                    }
                                }
                                property.comment = resultSet.getString("column_comment").replace("\"", "\\\"");
                                entity.properties.add(property);
                            }
                        }
                    }
                })
                .execute();
    }

    private void getByPostgres(FlowContext flowContext) {
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段信息")
                .putTemporaryData("sql", "select pg_class.relname as table_name, attname as column_name, attnum as oridinal_position, attnotnull as notnull, format_type(atttypid,atttypmod) as type, col_description(attrelid, attnum) as comment from pg_attribute join pg_class on pg_attribute.attrelid = pg_class.oid where attnum > 0 and atttypid > 0")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)){
                                    continue;
                                }
                                Property property = new Property();
                                property.column = resultSet.getString("column_name");
                                property.columnType = resultSet.getString("type");
                                property.notNull = "t".equals(resultSet.getString("notnull"));
                                property.comment = resultSet.getString("comment");
                                entity.properties.add(property);
                            }
                        }
                    }
                })
                .execute();

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段类型信息")
                .putTemporaryData("sql", "select table_name,ordinal_position,column_name,column_default,is_nullable,udt_name,character_maximum_length,column_default from information_schema.columns")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)) {
                                    continue;
                                }
                                String columnName =  resultSet.getString("column_name");
                                for (Property property : entity.properties) {
                                    if (!property.column.equalsIgnoreCase(columnName)) {
                                        continue;
                                    }
                                    property.columnType = resultSet.getString("udt_name");
                                    if (null != resultSet.getString("column_default")) {
                                        property.defaultValue = resultSet.getString("column_default");
                                        if(property.defaultValue.startsWith("nextval(")){
                                            property.id = true;
                                            property.strategy = IdStrategy.AutoIncrement;
                                        }
                                    }
                                }
                            }
                        }
                    }
                })
                .execute();
    }

    private void getBySQLServer(FlowContext flowContext){
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段类型信息")
                .putTemporaryData("sql", "select table_name,ordinal_position,column_name,column_type,is_nullable from information_schema.columns;")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)){
                                    continue;
                                }
                                Property property = new Property();
                                property.column = resultSet.getString("column_name");
                                property.columnType = resultSet.getString("column_type");
                                property.notNull = "NO".equals(resultSet.getString("is_nullable"));
                                entity.properties.add(property);
                            }
                        }
                    }
                })
                .execute();

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段注释")
                .putTemporaryData("sql", "select b.name table_name, c.name column_name, convert(varchar(255),a.value) value from sys.extended_properties a, sysobjects b, sys.columns c where a.major_id = b.id and c.object_id = b.id and c.column_id = a.minor_id;")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)) {
                                    continue;
                                }
                                String columnName =  resultSet.getString("column_name");
                                for (Property property : entity.properties) {
                                    if (!property.column.equalsIgnoreCase(columnName)) {
                                        continue;
                                    }
                                    property.comment = resultSet.getString("value");
                                }
                            }
                        }
                    }
                })
                .execute();

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表主键")
                .putTemporaryData("sql", "select table_name, column_name from information_schema.key_column_usage")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)) {
                                    continue;
                                }
                                String columnName =  resultSet.getString("column_name");
                                for (Property property : entity.properties) {
                                    if (!property.column.equalsIgnoreCase(columnName)) {
                                        continue;
                                    }
                                    property.id = true;
                                    property.strategy = IdStrategy.AutoIncrement;
                                }
                            }
                        }
                    }
                })
                .execute();
    }

    private void getByOracle(FlowContext flowContext) throws Exception {
        List<Entity> entityList = (List<Entity>) flowContext.checkData("entityList");

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段信息")
                .putTemporaryData("sql", "select table_name, column_name, column_type, nullable, data_length from user_tab_columns")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while (resultSet.next()) {
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)){
                                    continue;
                                }
                                Property property = new Property();
                                property.column = resultSet.getString("column_name");
                                property.columnType = resultSet.getString("column_type");
                                if(property.columnType.contains(" ")){
                                    property.columnType = property.columnType.substring(0,property.columnType.indexOf(" "));
                                }
                                String dataLength = resultSet.getString("data_length");
                                if(property.columnType.toLowerCase().contains("char")&&null!=dataLength&&!dataLength.isEmpty()){
                                    property.columnType += "(" + dataLength + ")";
                                }
                                property.notNull = "N".equals(resultSet.getString("nullable"));
                                entity.properties.add(property);
                            }
                        }
                    }
                })
                .execute();

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取表字段注释")
                .putTemporaryData("sql", "select table_name, column_name, comments from user_col_comments")
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String tableName = resultSet.getString("table_name");
                            for(Entity entity:entityList){
                                if(!entity.tableName.equalsIgnoreCase(tableName)) {
                                    continue;
                                }
                                String columnName =  resultSet.getString("column_name");
                                for (Property property : entity.properties) {
                                    if (!property.column.equalsIgnoreCase(columnName)) {
                                        continue;
                                    }
                                    property.comment = resultSet.getString("comments");
                                }
                            }
                        }
                    }
                })
                .execute();
    }
}
