package cn.schoolwow.quickdao.flow.ddl.index;

import cn.schoolwow.quickdao.annotation.IndexType;
import cn.schoolwow.quickdao.domain.external.IndexField;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.DatabaseType;
import cn.schoolwow.quickdao.domain.internal.common.ResultSetConsumer;
import cn.schoolwow.quickdao.flow.executor.ExecuteQueryConnectionFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class GetSingleEntityIndexFlow implements BusinessFlow {
    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        DatabaseType databaseType = (DatabaseType) flowContext.checkData("databaseType");

        flowContext.putTemporaryData("name", "获取指定表指定索引信息");
        switch (databaseType){
            case H2:{
                getByH2(flowContext);
            }break;
            case SQLite:{
                getBySQLite(flowContext);
            }break;
            case Mysql:
            case MariaDB:{
                getByMysql(flowContext);
            }break;
            case Postgresql:{
                getByPostgres(flowContext);
            }break;
            case SQLServer:{
                getBySQLServer(flowContext);
            }break;
            case Oracle:{
                getByOracle(flowContext);
            }break;
        }
    }

    @Override
    public String name() {
        return "获取数据库字段信息";
    }

    private void getByH2(FlowContext flowContext) {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select table_name, sql from information_schema.indexes where table_schema = '" + quickDAOConfig.databaseContext.databaseName + "' and table_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName.toUpperCase()))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while(resultSet.next()){
                            String sql = resultSet.getString("sql");
                            String[] tokens = sql.split("\"");
                            IndexField indexField = new IndexField();
                            if (tokens[0].contains("UNIQUE")) {
                                indexField.indexType = IndexType.UNIQUE;
                            } else {
                                indexField.indexType = IndexType.NORMAL;
                            }
                            indexField.indexName = tokens[3];
                            indexField.tableName = tokens[7];
                            for (int i = 9; i < tokens.length - 1; i++) {
                                indexField.columns.add(tokens[i]);
                            }
                            if(indexField.indexName.equalsIgnoreCase(indexName)){
                                flowContext.putData("indexField", indexField);
                                break;
                            }
                        }
                    }
                })
                .execute();
    }

    private void getBySQLite(FlowContext flowContext){
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select tbl_name, sql from sqlite_master where type='index' and tbl_name = ? and sql is not null;")
                .putTemporaryData("parameters", Arrays.asList(tableName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        List<String> warningSQLList = new ArrayList<>();
                        while (resultSet.next()) {
                            String sql = resultSet.getString("sql");
                            if (!sql.contains("\"") && !sql.contains("`")) {
                                warningSQLList.add(sql);
                                continue;
                            }
                            String[] tokens = sql.split("[\"|`]");
                            IndexField indexField = new IndexField();
                            if (tokens[0].contains("UNIQUE")) {
                                indexField.indexType = IndexType.UNIQUE;
                            } else {
                                indexField.indexType = IndexType.NORMAL;
                            }
                            indexField.indexName = tokens[1];
                            indexField.tableName = tokens[3];
                            for (int i = 5; i < tokens.length - 1; i++) {
                                indexField.columns.add(tokens[i]);
                            }
                            if(indexField.indexName.equalsIgnoreCase(indexName)){
                                flowContext.putData("indexField", indexField);
                                break;
                            }
                        }
                        if(!warningSQLList.isEmpty()){
                            flowContext.putTemporaryData("warningSQLList", warningSQLList);
                            flowContext.remark("存在异常索引!索引不包含\"也不包含`");
                        }
                    }
                })
                .execute();
    }

    private void getByMysql(FlowContext flowContext){
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select table_name, index_name, non_unique, column_name, index_type, index_comment from information_schema.`statistics` where table_schema = '" + quickDAOConfig.databaseContext.databaseName + "' and table_name = ? and index_name = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, indexName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        IndexField indexField = null;
                        while (resultSet.next()) {
                            if (null == indexField) {
                                indexField = new IndexField();
                                indexField.indexType = resultSet.getInt("non_unique") == 0 ? IndexType.UNIQUE : IndexType.NORMAL;
                                if ("FULLTEXT".equals(resultSet.getString("index_type"))) {
                                    indexField.indexType = IndexType.FULLTEXT;
                                }
                                indexField.indexName = resultSet.getString("index_name");
                                indexField.columns.add(resultSet.getString("column_name"));
                                indexField.using = resultSet.getString("index_type");
                                indexField.comment = resultSet.getString("index_comment");
                            } else {
                                indexField.columns.add(resultSet.getString("column_name"));
                            }
                        }
                        flowContext.putData("indexField", indexField);
                    }
                })
                .execute();
    }

    private void getByPostgres(FlowContext flowContext){
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select tablename,indexname,indexdef from pg_indexes where tablename = ? and indexname = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, indexName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            IndexField indexField = new IndexField();
                            indexField.tableName = resultSet.getString("tablename");
                            indexField.indexName = resultSet.getString("indexname");

                            String def = resultSet.getString("indexdef");
                            if (def.contains("UNIQUE INDEX")) {
                                indexField.indexType = IndexType.UNIQUE;
                            } else {
                                indexField.indexType = IndexType.NORMAL;
                            }
                            indexField.using = def.substring(def.indexOf("USING") + "USING".length(), def.indexOf("(")).replace("\"", "").trim();
                            String[] columns = def.substring(def.indexOf("(") + 1, def.indexOf(")")).split(",");
                            for (int i = 0; i < columns.length; i++) {
                                indexField.columns.add(columns[i].trim());
                            }
                            flowContext.putData("indexField", indexField);
                        }
                    }
                })
                .execute();
    }

    private void getBySQLServer(FlowContext flowContext){
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select i.is_unique,i.name,col.name col_name from sys.indexes i left join sys.index_columns ic on ic.object_id = i.object_id and ic.index_id = i.index_id left join (select * from sys.all_columns where object_id = object_id(?, N'U' )) col on ic.column_id = col.column_id where i.object_id = object_id(?, N'U' ) and i.index_id > 0 and i.name = ?;")
                .putTemporaryData("parameters", Arrays.asList(tableName, tableName, indexName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        IndexField indexField = null;
                        while(resultSet.next()){
                            if(null==indexField){
                                indexField = new IndexField();
                                if (resultSet.getBoolean("is_unique")) {
                                    indexField.indexType = IndexType.UNIQUE;
                                } else {
                                    indexField.indexType = IndexType.NORMAL;
                                }
                                indexField.indexName = resultSet.getString("name");
                                indexField.columns.add(resultSet.getNString("col_name"));
                            }else{
                                indexField.columns.add(resultSet.getNString("col_name"));
                            }
                        }
                        flowContext.putData("indexField", indexField);
                    }
                })
                .execute();
    }

    private void getByOracle(FlowContext flowContext){
        String tableName = (String) flowContext.checkData("tableName");
        String indexName = (String) flowContext.checkData("indexName");
        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("sql", "select table_name, index_name,uniqueness from user_indexes where table_name = ? and index_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName,indexName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        if(resultSet.next()){
                            IndexField indexField = new IndexField();
                            indexField.tableName = resultSet.getString("table_name");
                            if("UNIQUE".equalsIgnoreCase(resultSet.getString("uniqueness"))){
                                indexField.indexType = IndexType.UNIQUE;
                            }else{
                                indexField.indexType = IndexType.NORMAL;
                            }
                            indexField.indexName = resultSet.getString("index_name");
                            flowContext.putData("indexField", indexField);
                        }
                    }
                })
                .execute();

        IndexField indexField = (IndexField) flowContext.getData("indexField");
        if(null==indexField){
            return;
        }

        flowContext.startFlow(new ExecuteQueryConnectionFlow())
                .putTemporaryData("name", "获取索引字段信息")
                .putTemporaryData("sql", "select table_name,index_name,column_name from user_ind_columns where table_name = ? and index_name = ?")
                .putTemporaryData("parameters", Arrays.asList(tableName, indexName))
                .putReturnData("resultSetConsumer",new ResultSetConsumer() {
                    @Override
                    public void consumeResultSet(ResultSet resultSet) throws Exception {
                        while (resultSet.next()) {
                            String columnName = resultSet.getString("column_name");
                            indexField.columns.add(columnName);
                        }
                    }
                })
                .execute();
    }
}
