package cn.schoolwow.quickdao.domain.external;

import cn.schoolwow.quickdao.dao.DAO;
import cn.schoolwow.quickdao.dao.transaction.Transaction;
import cn.schoolwow.quickdao.domain.internal.config.*;
import cn.schoolwow.quickdao.flow.ddl.index.GetEntityIndexFlow;
import cn.schoolwow.quickdao.flow.ddl.property.GetEntityPropertyFlow;
import cn.schoolwow.quickdao.flow.ddl.table.GetEntityFlow;
import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;


/**
 * 数据源访问配置选项
 */
public class QuickDAOConfig {
    private Logger logger = LoggerFactory.getLogger(QuickDAOConfig.class);

    /**
     * 实体类选项
     */
    public EntityOption entityOption = new EntityOption();

    /**
     * 数据库选项
     */
    public DatabaseOption databaseOption = new DatabaseOption();

    /**
     * 数据库上下文
     */
    public DatabaseContext databaseContext = new DatabaseContext();

    /**
     * 日志记录选项
     * */
    public LogRecordOption logRecordOption = new LogRecordOption();

    /**
     * 日志记录选项
     * */
    public UpgradeOption upgradeOption = new UpgradeOption();

    /**
     * 线程绑定事务
     * */
    public ThreadLocal<Transaction> transactionThreadLocal = new ThreadLocal<>();

    /**
     * 异步任务线程池
     * */
    public ThreadPoolExecutor asyncThreadPool = (ThreadPoolExecutor) Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());

    public QuickFlow quickFlow;

    public DAO dao;

    /**
     * 根据类名获取实体类信息
     */
    public Entity getEntityByClassName(String className) {
        if (databaseContext.entityMap.containsKey(className)) {
            return databaseContext.entityMap.get(className);
        }
        throw new IllegalArgumentException("扫描实体类列表中不包含该实体类!类名:" + className);
    }

    /**
     * 根据表名获取数据库信息
     */
    public Entity getDatabaseEntityByTableName(String tableName) {
        if (!databaseContext.databaseTableCache.containsKey(tableName)) {
            FlowContext flowContext = quickFlow.startFlow("获取数据库指定表信息")
                    .putTemporaryData("tableName", tableName)
                    .putTemporaryData("insertSqlLog", false)
                    .next(new GetEntityFlow())
                    .next(new GetEntityPropertyFlow())
                    .next(new GetEntityIndexFlow())
                    .execute();
            Entity entity = (Entity) flowContext.getData("entity");
            if (null == entity) {
                return null;
            }
            databaseContext.databaseTableCache.put(tableName, entity);
        }
        return databaseContext.databaseTableCache.get(tableName);
    }

    /**
     * 删除缓存数据表信息
     */
    public void deleteDatabaseEntityCache(String tableName) {
        Iterator<Map.Entry<String, String>> entryIterator = databaseContext.statementCache.entrySet().iterator();
        while (entryIterator.hasNext()) {
            Map.Entry<String, String> entry = entryIterator.next();
            if (entry.getKey().contains("_" + tableName + "_")) {
                logger.trace("删除SQL语句缓存:{}", entry.getKey());
                entryIterator.remove();
            }
        }
        logger.trace("删除数据库表缓存,表名:{}", tableName);
        databaseContext.databaseTableCache.remove(tableName);
    }

}
