package cn.schoolwow.quickdao.dao.operation;

import cn.schoolwow.quickdao.dao.transaction.Transaction;
import cn.schoolwow.quickdao.dao.transaction.TransactionInvocationHandler;
import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.dml.ManipulationOption;
import cn.schoolwow.quickdao.exception.SQLRuntimeException;
import cn.schoolwow.quickdao.flow.dml.instance.insert.InsertInstanceArrayFlow;
import cn.schoolwow.quickdao.provider.DatabaseProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

public class DAOOperationImpl implements DAOOperation {
    private Logger logger = LoggerFactory.getLogger(DAOOperationImpl.class);

    private final QuickDAOConfig quickDAOConfig;

    public DAOOperationImpl(QuickDAOConfig quickDAOConfig) {
        this.quickDAOConfig = quickDAOConfig;
    }

    @Override
    public Transaction startTransaction() {
        TransactionInvocationHandler transactionInvocationHandler = new TransactionInvocationHandler(quickDAOConfig);
        Transaction transactionProxy = (Transaction) Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class<?>[]{Transaction.class}, transactionInvocationHandler);
        return transactionProxy;
    }

    @Override
    public void startTransaction(Consumer<Transaction> transactionConsumer) {
        Transaction transaction = startTransaction();
        Connection connection = transaction.getTransactionConnection();
        try {
            transactionConsumer.accept(transaction);
            connection.commit();
        } catch (Exception e) {
            try {
                connection.rollback();
            } catch (SQLException ex) {
                ex.printStackTrace();
            }
            throw new SQLRuntimeException(e);
        } finally {
            try {
                connection.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public DataSource getDataSource() {
        return quickDAOConfig.databaseContext.dataSource;
    }

    @Override
    public Map<String, Entity> getEntityMap() {
        return quickDAOConfig.databaseContext.entityMap;
    }

    @Override
    public Entity getEntity(Class clazz) {
        return quickDAOConfig.databaseContext.entityMap.values().stream().filter(entity -> entity.clazz.getName().equalsIgnoreCase(clazz.getName())).findFirst().orElse(null);
    }

    @Override
    public Entity getEntity(String tableName) {
        return quickDAOConfig.databaseContext.entityMap.values().stream().filter(entity -> entity.tableName.equalsIgnoreCase(tableName)).findFirst().orElse(null);
    }

    @Override
    public Property getEntityProperty(Class clazz, String fieldName) {
        Entity entity = getEntity(clazz);
        if(null==entity){
            throw new IllegalArgumentException("未扫描到实体类!实体类名:"+clazz.getName());
        }
        return entity.getPropertyByFieldNameOrColumnName(fieldName);
    }

    @Override
    public DatabaseProvider getDatabaseProvider() {
        return quickDAOConfig.databaseContext.databaseProvider;
    }

    @Override
    public QuickDAOConfig getQuickDAOConfig() {
        return quickDAOConfig;
    }

    @Override
    public void recordFlowLog(boolean recordFlowLog) {
        quickDAOConfig.logRecordOption.recordFlowLog = recordFlowLog;
    }

    @Override
    public void recordSqlLog(boolean recordSqlLog) {
        quickDAOConfig.logRecordOption.recordSqlLog = recordSqlLog;
    }

    @Override
    public void startRecord() {
        quickDAOConfig.logRecordOption.record = true;
    }

    @Override
    public String stopRecord() {
        quickDAOConfig.logRecordOption.record = false;
        String sqlRecord = quickDAOConfig.logRecordOption.sqlRecordBuilder.toString();
        quickDAOConfig.logRecordOption.sqlRecordBuilder.setLength(0);
        return sqlRecord;
    }

    @Override
    public <T> T log(Class<T> logEntity) {
        return logMap("#SINGLE_INSTANCE#", logEntity);
    }

    @Override
    public <T> T logIfExist(Class<T> logEntity) {
        return logMapIfExist("#SINGLE_INSTANCE#", logEntity);
    }

    @Override
    public <T> T logMapIfExist(String uniqueKey, Class<T> logEntity) {
        if(null==uniqueKey||uniqueKey.isEmpty()){
            throw new IllegalArgumentException("唯一键不能为空");
        }
        try {
            if(null==quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(classNameLogInstance.containsKey(uniqueKey)){
                T logInstance = (T) classNameLogInstance.get(uniqueKey);
                return logInstance;
            }
            return null;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public <T> List<T> logMap(Class<T> logEntity) {
        try {
            if(null==quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(null==classNameLogInstance){
                return null;
            }
            List<T> instanceList = new ArrayList<>();
            for(Object instance:classNameLogInstance.values()){
                instanceList.add((T) instance);
            }
            return instanceList;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public <T> T logMap(String uniqueKey, Class<T> logEntity) {
        if(null==uniqueKey||uniqueKey.isEmpty()){
            throw new IllegalArgumentException("唯一键不能为空");
        }
        try {
            if(null==quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(!classNameLogInstance.containsKey(uniqueKey)){
                classNameLogInstance.put(uniqueKey, logEntity.newInstance());
            }
            T logInstance = (T) classNameLogInstance.get(uniqueKey);
            return logInstance;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public void clearLog() {
        Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get();
        if(null!=classNameLogInstanceMap){
            classNameLogInstanceMap.clear();
        }
    }

    @Override
    public void insertLog() {
        if(null==quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get()){
            Map<String,Map<String,Object>> classNameLogInstanceMap = new ConcurrentHashMap<>();
            quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
        }
        Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.logRecordOption.classNameLogInstanceMapThreadLocal.get();
        if(null==classNameLogInstanceMap){
            return;
        }
        Collection<Map<String,Object>> logInstanceMapCollection = classNameLogInstanceMap.values();
        for(Map<String,Object> logInstanceMap:logInstanceMapCollection){
            Collection<Object> logInstances = logInstanceMap.values();
            quickDAOConfig.quickFlow.startFlow(new InsertInstanceArrayFlow())
                    .putTemporaryData("instances", logInstances.toArray(new Object[0]))
                    .putTemporaryData("manipulationOption", new ManipulationOption())
                    .execute();
        }
        classNameLogInstanceMap.clear();
    }

}