package cn.schoolwow.quickdao.dao.dql.condition;

import cn.schoolwow.quickdao.dao.dql.response.Response;
import cn.schoolwow.quickdao.dao.dql.subCondition.SubCondition;
import cn.schoolwow.quickdao.domain.external.dql.QueryColumnTypeMapping;

import java.io.Serializable;
import java.util.Collection;

/**
 * 查询接口
 *
 * <p>本接口定义单表查询接口,多表关联查询等方法</p>
 *
 * <p>对于查询接口您需要知道以下几点</p>
 * <ol>
 *     <li>实体类字段使用驼峰式命名映射到数据库中.例如字段<b>firstName</b>映射到数据库后字段名为<b>first_name</b>.
 *     <br/><b>如您使用了@ColumnName注解则以该注解设置的字段为准</b>
 *     </li>
 *     <li>对于关联查询,主表别名为t,使用join方法添加的子表按照添加顺序依次为t1,t2,t3......
 *     <br/><b>如您使用了tableAliasName方法则以该方法设置的值为准</b>
 *     <p></p>
 *     <p>关联查询默认关联主表中唯一一个类型为子表的成员变量.</p>
 *     <pre>
 *         Parent{
 *             long id;
 *         }
 *         Child{
 *             long id;
 *             long parentId;
 *             Parent parent;
 *         }
 *         //关联Child类的唯一一个子表成员变量的parent属性
 *         joinTable(Parent.class,"parentId","id").compositField();
 *     </pre>
 *     <p>若出现多个相同的子表成员变量,必须手动指定相应成员变量</p>
 *     <pre>
 *         Parent{
 *             long id;
 *         }
 *         Child{
 *             long id;
 *             long fatherId;
 *             long motherId;
 *
 *             Parent father;
 *             Parent mother;
 *         }
 *         //关联多个时需要手动指定
 *         joinTable(Parent.class,"fatherId","id","father").compositField();
 *         joinTable(Parent.class,"motherId","id","mother").compositField();
 *     </pre>
 *     <p>若需返回关联实体字段,必须调用<b>compositField</b>方法</p>
 *     </li>
 *     <li>当您调用getArray方法,返回字段名称均为<b>表别名_字段名</b>.例如主表中的firstName字段,则返回字段名称为<b>t_first_name</b></li>
 *     <li>对于直接添加查询语句的方法,请注意查询语句参数将直接拼接到sql字符上,不会做任何转义操作,请注意SQL注入等安全问题</li>
 * </ol>
 * </p>
 */
public interface Condition<T> extends Serializable {
    /**
     * 设置主表别名
     *
     * @param tableAliasName 主表别名
     */
    Condition<T> tableAliasName(String tableAliasName);

    /**
     * 添加distinct关键字
     */
    Condition<T> distinct();

    /**
     * 添加空查询
     *
     * @param field 指明哪个字段为Null
     */
    Condition<T> addNullQuery(String field);

    /**
     * 添加非空查询
     *
     * @param field 指明哪个字段不为Null
     */
    Condition<T> addNotNullQuery(String field);

    /**
     * 添加空查询
     *
     * @param field 指明哪个字段不为空字符串
     */
    Condition<T> addEmptyQuery(String field);

    /**
     * 添加非空查询
     *
     * @param field 指明哪个字段不为空字符串
     */
    Condition<T> addNotEmptyQuery(String field);

    /**
     * 添加范围查询语句
     *
     * @param field   字段名
     * @param inQuery 英文逗号隔开的字段值
     */
    Condition<T> addInQuery(String field, String inQuery);

    /**
     * 添加范围查询语句
     *
     * @param field  字段名
     * @param values 指明在该范围内的值
     */
    Condition<T> addInQuery(String field, Object... values);

    /**
     * 添加范围查询语句
     *
     * @param field  字段名
     * @param values 指明在该范围内的值
     */
    Condition<T> addInQuery(String field, Collection values);

    /**
     * 添加范围查询语句
     *
     * @param field   字段名
     * @param inQuery 英文逗号隔开的字段值
     */
    Condition<T> addNotInQuery(String field, String inQuery);

    /**
     * 添加范围查询语句
     *
     * @param field  字段名
     * @param values 指明在不该范围内的值
     */
    Condition<T> addNotInQuery(String field, Object... values);

    /**
     * 添加范围查询语句
     *
     * @param field  字段名
     * @param values 指明在不该范围内的值
     */
    Condition<T> addNotInQuery(String field, Collection values);

    /**
     * 添加between语句
     *
     * @param field 字段名
     * @param start 范围开始值
     * @param end   范围结束值
     */
    Condition<T> addBetweenQuery(String field, Object start, Object end);

    /**
     * 添加Like查询
     *
     * @param field 字段名
     * @param value 字段值
     */
    Condition<T> addLikeQuery(String field, Object value);

    /**
     * 添加Not Like查询
     *
     * @param field 字段名
     * @param value 字段值
     */
    Condition<T> addNotLikeQuery(String field, Object value);

    /**
     * 添加字段查询
     *
     * @param field 字段名
     * @param value 字段值
     */
    Condition<T> addQuery(String field, Object value);

    /**
     * 添加字段查询
     *
     * @param field    字段名
     * @param operator 操作符,可为<b>></b>,<b>>=</b>,<b>=</b>,<b><</b><b><=</b>
     * @param value    字段值
     */
    Condition<T> addQuery(String field, String operator, Object value);

    /**
     * 添加开闭区间字段查询
     *
     * @param field    字段名
     * @param value 区间值,例如(1,100)或者[1,100]
     */
    Condition<T> addIntervalQuery(String field, String value);

    /**
     * 添加复杂集合查询
     *
     * <pre>
     * 支持语法
     * empty: 空查询
     * not empty: 非空查询
     * like: like查询
     * not like: not like查询
     * in: in查询
     * not in: not in 查询
     * interval: 开闭区间查询
     * between: 范围查询
     * gt: 大于
     * gte: 大于等于
     * lt: 小于
     * lte: 小于等于
     * </pre>
     *
     * @param field    字段名
     * @param value 根据语法设置对应查询请求
     */
    Condition<T> addCompositeQuery(String field, String value);

    /**
     * 自定义查询语句,具体映射规则请看此{@link cn.schoolwow.quickdao.dao.dql.condition.Condition}
     *
     * @param query         子查询条件,可使用?占位符
     * @param parameterList 占位符参数列表,可为null
     */
    Condition<T> addRawQuery(String query, Object... parameterList);

    /**
     * 添加where子查询
     *
     * @param field    字段名
     * @param subQuery 子查询语句
     */
    Condition<T> addSubQuery(String field, Condition subQuery);

    /**
     * 添加where子查询
     *
     * @param field    字段名
     * @param operator 操作符,可为<b>></b>,<b>>=</b>,<b>=</b>,<b><</b><b><=</b>
     * @param subQuery 子查询语句
     */
    Condition<T> addSubQuery(String field, String operator, Condition subQuery);

    /**
     * 添加exist子查询
     *
     * @param subQuery 子查询语句
     */
    Condition<T> addExistSubQuery(Condition subQuery);

    /**
     * 添加not exist子查询
     *
     * @param subQuery 子查询语句
     */
    Condition<T> addNotExistSubQuery(Condition subQuery);

    /**
     * 添加自定义字段,具体映射规则请看Condition类的JavaDoc注释
     *
     * @param fields 自定义查询列
     */
    Condition<T> addColumn(String... fields);

    /**
     * 添加主表字段
     */
    Condition<T> addTableColumn();

    /**
     * 排除指定字段
     *
     * @param excludeFields 指定主表排除字段
     */
    Condition<T> addExcludeColumn(String... excludeFields);

    /**
     * 设置返回列类型转换
     *
     * @param queryColumnTypeMapping 列类型转换函数,参数为列信息,返回列的类型
     */
    Condition<T> setColumnTypeMapping(QueryColumnTypeMapping queryColumnTypeMapping);

    /**
     * 设置返回列类型转换
     * @param columnLabel 返回列名
     * @param type 类型
     */
    Condition<T> setColumnTypeMap(String columnLabel, Class type);

    /**
     * 添加select子查询
     *
     * @param subQuery 子查询
     */
    Condition<T> addColumn(Condition subQuery);

    /**
     * 添加select子查询
     *
     * @param subQuery        子查询
     * @param columnNameAlias 指定子查询返回列名的别名
     */
    Condition<T> addColumn(Condition subQuery, String columnNameAlias);

    /**
     * 添加更新字段,用于{@link cn.schoolwow.quickdao.dao.dql.response.Response#update()}方法
     *
     * @param field 待更新的字段
     * @param value 待更新字段的值
     */
    Condition<T> addUpdate(String field, Object value);

    /**
     * 自定义更新语句
     *
     * @param update        更新语句,可使用?占位符
     * @param parameterList 占位符参数列表,可为null
     */
    Condition<T> addRawUpdate(String update, Object... parameterList);

    /**
     * 添加or查询,具体映射规则请看此{@link cn.schoolwow.quickdao.dao.dql.condition.Condition}
     *
     * @param or            or查询语句,可使用?占位符
     * @param parameterList 占位符参数值,无参数时可为null
     */
    Condition<T> or(String or, Object... parameterList);

    /**
     * 添加分组查询
     *
     * @param fields 分组字段
     */
    Condition<T> groupBy(String... fields);

    /**
     * 添加having查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param having        having查询子句,可使用?占位符
     * @param parameterList 占位符参数值
     * @see Condition
     */
    Condition<T> having(String having, Object... parameterList);

    /**
     * 添加having子查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param field    字段名
     * @param operator 操作符,可为<b>></b>,<b>>=</b>,<b>=</b>,<b><</b><b><=</b>
     * @param subQuery 子查询语句
     */
    Condition<T> having(String field, String operator, Condition subQuery);

    /**
     * cross join 关联表,获取笛卡尔积
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param clazz 待关联的子表
     */
    <E> SubCondition<E, T> crossJoinTable(Class<E> clazz);

    /**
     * cross join 关联表,获取笛卡尔积
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param tableName 待关联的子表
     */
    SubCondition<?, T> crossJoinTable(String tableName);

    /**
     * 关联表查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param clazz          待关联的子表
     * @param primaryField   <b>主表</b>关联字段
     * @param joinTableField <b>子表</b>关联字段
     */
    <E> SubCondition<E, T> joinTable(Class<E> clazz, String primaryField, String joinTableField);

    /**
     * 关联表查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     *
     * @param clazz          待关联的子表
     * @param primaryField   <b>主表</b>关联字段
     * @param joinTableField <b>子表</b>关联字段
     * @param compositField  <b>子表</b>实体类成员变量名
     */
    <E> SubCondition<E, T> joinTable(Class<E> clazz, String primaryField, String joinTableField, String compositField);

    /**
     * 关联子查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     * <p>调用本方法将在sql语句中拼接如下字符串</p>
     * <p><b>join #{condition子表} as t1 on t.primaryField = t1.joinTableField</b>
     * </p>
     *
     * @param joinCondition      关联Condition
     * @param primaryField       <b>主表</b>关联字段
     * @param joinConditionField <b>子查询</b>关联字段
     */
    <E> SubCondition<E, T> joinTable(Condition<E> joinCondition, String primaryField, String joinConditionField);

    /**
     * 关联表查询
     * <p>调用本方法时请先查看Condition类JavaDoc注释</p>
     * <p>调用本方法将在sql语句中拼接如下字符串</p>
     * <p><b>join #{clazz} as t1 on t.primaryField = t1.joinTableField</b>
     * </p>
     *
     * @param tableName      待关联的子表
     * @param primaryField   <b>主表</b>关联字段
     * @param joinTableField <b>子表</b>关联字段
     */
    SubCondition<?, T> joinTable(String tableName, String primaryField, String joinTableField);

    /**
     * 设置指定字段排序方式
     *
     * @param field 升序排列字段名
     * @param asc   排序方式,值为asc或者desc
     */
    Condition<T> order(String field, String asc);

    /**
     * 根据指定字段升序排列
     *
     * @param field 升序排列字段名
     */
    Condition<T> orderBy(String... field);

    /**
     * 根据指定字段降序排列
     *
     * @param field 降序排列字段名
     */
    Condition<T> orderByDesc(String... field);

    /**
     * 手动指定排序
     *
     * @param orderBy 排序
     */
    Condition<T> orderByRaw(String orderBy);

    /**
     * 分页操作
     * <b>请注意,oracle数据库调用分页方法后请直接执行execute方法</b>
     *
     * @param offset 偏移量
     * @param limit  返回个数
     */
    Condition<T> limit(long offset, long limit);

    /**
     * 分页操作
     * <b>请注意,oracle数据库调用分页方法后请直接执行execute方法</b>
     *
     * @param pageNum  第几页
     * @param pageSize 每页个数
     */
    Condition<T> page(int pageNum, int pageSize);

    /**
     * 返回子表实体类字段信息
     *
     * @see {@link cn.schoolwow.quickdao.dao.dql.condition.Condition#joinTable(Class, String, String)} ()}
     * @see {@link cn.schoolwow.quickdao.dao.dql.condition.Condition#joinTable(Class, String, String, String)} ()}
     */
    Condition<T> compositeField();

    /**
     * 获取lambdaCondition对象
     */
    LambdaCondition<T> lambdaCondition();

    /**
     * 执行并返回Response实例
     */
    Response<T> execute();

    /**
     * 执行并返回Response实例
     */
    Condition<T> clone();

}
