package cn.schoolwow.quickdao.flow.executor;

import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.dml.GetBatchParametersSupplier;
import cn.schoolwow.quickdao.flow.executor.common.BeforeExecuteFlow;
import cn.schoolwow.quickdao.flow.executor.common.ExecuteFailFlow;
import cn.schoolwow.quickdao.flow.executor.common.InsertSqlLogFlow;
import cn.schoolwow.quickdao.flow.executor.common.SetPrepareStatementParameterFlow;
import cn.schoolwow.quickdao.util.ParametersUtil;
import cn.schoolwow.quickflow.domain.FlowContext;
import cn.schoolwow.quickflow.flow.BusinessFlow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

public class ExecuteBatchUpdateConnectionFlow implements BusinessFlow {
    private Logger logger = LoggerFactory.getLogger(ExecuteBatchUpdateConnectionFlow.class);

    @Override
    public void executeBusinessFlow(FlowContext flowContext) throws Exception {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        String name = (String) flowContext.checkData("name");
        String sql = (String) flowContext.checkData("sql");
        Connection transactionConnection = (Connection) flowContext.getData("transactionConnection");

        flowContext.executeFlowList(new BeforeExecuteFlow());
        logger.debug("[批处理更新]名称:{}, 执行语句:{}", name, sql);

        long startTime = System.currentTimeMillis();
        try {
            if(null==transactionConnection){
                try (Connection connection = quickDAOConfig.databaseContext.dataSource.getConnection()){
                    executeBatchUpdate(connection, flowContext);
                }
            }else{
                executeBatchUpdate(transactionConnection, flowContext);
            }
        }catch (SQLException e){
            throw e;
        }finally {
            flowContext.removeData("getBatchParametersSupplier");
            if(quickDAOConfig.logRecordOption.record&&!"插入SQL日志".equalsIgnoreCase(name)){
                long endTime = System.currentTimeMillis();
                long consumeTime = endTime-startTime;
                quickDAOConfig.logRecordOption.sqlRecordBuilder.append("[更新]名称:"+name+",耗时:"+consumeTime+"ms,执行语句:"+sql+"\r\n");
            }
        }
    }

    @Override
    public String name() {
        return "执行批处理更新语句";
    }

    private void executeBatchUpdate(Connection connection, FlowContext flowContext) throws Exception {
        QuickDAOConfig quickDAOConfig = (QuickDAOConfig) flowContext.checkData("quickDAOConfig");
        Integer perBatchCount = (Integer) flowContext.getData("perBatchCount", quickDAOConfig.databaseOption.perBatchCount);
        String sql = (String) flowContext.checkData("sql");
        Integer size = (Integer) flowContext.checkData("size");
        GetBatchParametersSupplier getBatchParametersSupplier = (GetBatchParametersSupplier) flowContext.useData("getBatchParametersSupplier");

        List<Object> parameters = null;
        int effect = 0;
        flowContext.putTemporaryData("startTime", System.currentTimeMillis());
        try (PreparedStatement batchPrepareStatement = connection.prepareStatement(sql);){
            for (int i = 0; i < size; i += perBatchCount) {
                batchPrepareStatement.clearBatch();
                int end = Math.min(i + perBatchCount, size);
                for (int j = i; j < end; j++) {
                    parameters = getBatchParametersSupplier.getBatchParameters(j);
                    String currentFormatSQL = ParametersUtil.getFormatSQL(sql, parameters);
                    flowContext.putTemporaryData("formatSQL", currentFormatSQL);
                    flowContext.startFlow(new SetPrepareStatementParameterFlow())
                            .putTemporaryData("preparedStatement", batchPrepareStatement)
                            .putTemporaryData("parameters", parameters)
                            .execute();
                    batchPrepareStatement.addBatch();
                }
                effect += executeBatch(batchPrepareStatement);
            }
            flowContext.startFlow(new InsertSqlLogFlow())
                    .putTemporaryData("sqlLogType", 1)
                    .execute();
        }catch (Exception e){
            flowContext.startFlow(new InsertSqlLogFlow())
                    .next(new ExecuteFailFlow())
                    .putTemporaryData("exception", e)
                    .putTemporaryData("sqlLogType", 1)
                    .execute();
            throw e;
        }finally {
            flowContext.putData("effect", effect);
        }
    }

    private int executeBatch(PreparedStatement batchPrepareStatement) throws SQLException {
        int effect = 0;
        int[] batches = batchPrepareStatement.executeBatch();
        for (int batch : batches) {
            switch (batch) {
                case Statement.SUCCESS_NO_INFO: {
                    effect += 1;
                }
                break;
                case Statement.EXECUTE_FAILED: {
                }
                break;
                default: {
                    effect += batch;
                }
            }
        }
        return effect;
    }

}
