package cn.schoolwow.quickdao.dao.operation;

import cn.schoolwow.quickdao.dao.transaction.Transaction;
import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.provider.DatabaseProvider;

import javax.sql.DataSource;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

public interface DAOOperation {

    /**
     * 开启事务
     */
    Transaction startTransaction();

    /**
     * 开启事务
     */
    void startTransaction(Consumer<Transaction> transactionConsumer);

    /**
     * 获取连接池
     */
    DataSource getDataSource();

    /**
     * 获取扫描的所有实体类信息
     */
    Map<String, Entity> getEntityMap();

    /**
     * 获取实体类表
     *
     * @param clazz 实体类
     */
    Entity getEntity(Class clazz);

    /**
     * 获取实体类表
     *
     * @param tableName 数据库表名
     */
    Entity getEntity(String tableName);

    /**
     * 获取实体类表字段
     *
     * @param clazz 实体类
     * @param fieldName 字段名
     */
    Property getEntityProperty(Class clazz, String fieldName);

    /**
     * 获取数据库提供者
     */
    DatabaseProvider getDatabaseProvider();

    /**
     * 获取配置信息
     */
    QuickDAOConfig getQuickDAOConfig();

    /**
     * 是否记录流程日志
     * */
    void recordFlowLog(boolean recordFlowLog);

    /**
     * 是否记录SQL日志
     * */
    void recordSqlLog(boolean recordSqlLog);

    /**
     * 开启记录模式,只记录SQL语句,不执行
     * */
    void startRecord();

    /**
     * 停止记录模式
     * @return 返回记录SQL语句
     * */
    String stopRecord();

    /**
     * 线程绑定日志记录,总是创建新日志对象
     * @param logEntity 日志实例类
     * */
    <T> T log(Class<T> logEntity);

    /**
     * 线程绑定日志记录,存在则返回,不存在返回null
     * @param logEntity 日志实例类
     * */
    <T> T logIfExist(Class<T> logEntity);

    /**
     * 线程绑定日志实例记录,存在则返回,不存在返回null
     * @param uniqueKey 日志实例唯一值
     * @param logEntity 日志实例类
     * */
    <T> T logMapIfExist(String uniqueKey, Class<T> logEntity);

    /**
     * 线程绑定多个日志实例记录,总是创建新日志对象
     * @param logEntity 日志实例类
     * */
    <T> List<T> logMap(Class<T> logEntity);

    /**
     * 线程绑定多个日志实例记录,总是创建新日志对象
     * @param uniqueKey 日志实例唯一值
     * @param logEntity 日志实例类
     * */
    <T> T logMap(String uniqueKey, Class<T> logEntity);

    /**
     * 清除日志记录
     * */
    void clearLog();

    /**
     * 插入线程绑定日志记录
     * */
    void insertLog();
}
