package cn.schoolwow.quickdao.dao.ddl;

import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.IndexField;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.domain.external.PropertyOption;

import java.util.List;

/**
 * 数据定义语言
 */
public interface DatabaseDefinition {
    /**
     * 表是否存在
     *
     * @param clazz 实体类
     */
    boolean hasTable(Class clazz);

    /**
     * 表是否存在
     *
     * @param tableName 表名
     */
    boolean hasTable(String tableName);

    /**
     * 列是否存在
     *
     * @param clazz 实体类
     * @param columnName 列名
     */
    boolean hasColumn(Class clazz, String columnName);

    /**
     * 列是否存在
     *
     * @param tableName  表名
     * @param columnName 列名
     */
    boolean hasColumn(String tableName, String columnName);

    /**
     * 获取数据库所有表名
     */
    List<String> getTableNameList();

    /**
     * 获取数据库表列表
     */
    List<Entity> getDatabaseEntityList();

    /**
     * 获取数据库表列表
     *
     * @param tableName 表名
     */
    Entity getDatabaseEntity(String tableName);

    /**
     * 获取表字段列表
     *
     * @param clazz 实体类
     */
    List<Property> getPropertyList(Class clazz);

    /**
     * 获取表字段列表
     *
     * @param tableName 表名
     */
    List<Property> getPropertyList(String tableName);

    /**
     * 获取表字段
     *
     * @param clazz      实体类
     * @param columnName 字段名称
     */
    Property getProperty(Class clazz, String columnName);

    /**
     * 获取表指定字段
     *
     * @param tableName  表名
     * @param columnName 字段名称
     */
    Property getProperty(String tableName, String columnName);

    /**
     * 建表
     */
    void create(Class clazz);

    /**
     * 建表
     */
    void create(Entity entity);

    /**
     * 重命名表
     */
    void renameTable(String oldTableName, String newTableName);

    /**
     * 删表
     */
    void dropTable(Class clazz);

    /**
     * 删表
     */
    void dropTable(String tableName);

    /**
     * 重建表
     */
    void rebuild(Class clazz);

    /**
     * 重建表
     */
    void rebuild(String tableName);

    /**
     * 新增列
     *
     * @param tableName 表名
     * @param propertyOption  字段属性
     * @return 修改列
     */
    Property createColumn(String tableName, PropertyOption propertyOption);

    /**
     * 重命名列
     */
    void renameTableColumn(String tableName, String oldTableColumn, String newTableColumn);

    /**
     * 修改列
     * @param tableName 表名
     * @param propertyOption 列信息
     */
    void alterColumn(String tableName, PropertyOption propertyOption);

    /**
     * 删除列
     *
     * @param tableName  表名
     * @param columnName 列名
     */
    Property dropColumn(String tableName, String columnName);

    /**
     * 索引是否存在
     *
     * @param tableName 表名
     * @param indexName 索引名称
     */
    boolean hasIndex(String tableName, String indexName);

    /**
     * 约束否存在
     *
     * @param tableName      表名
     * @param constraintName 约束名称
     */
    boolean hasConstraint(String tableName, String constraintName);

    /**
     * 获取索引列表
     *
     * @param tableName 表名
     */
    List<IndexField> getIndexFieldList(String tableName);

    /**
     * 获取指定索引
     *
     * @param tableName 表名
     */
    IndexField getIndexField(String tableName, String indexName);

    /**
     * 新增索引
     *
     * @param indexField 索引信息
     */
    void createIndex(IndexField indexField);

    /**
     * 删除索引
     *
     * @param tableName 表名
     * @param indexName 索引名称
     */
    void dropIndex(String tableName, String indexName);

    /**
     * 创建外键约束
     *
     * @param tableName 表名
     * @param propertyOption 外键信息
     */
    void createForeignKey(String tableName, PropertyOption propertyOption);

    /**
     * 是否开启外键约束检查
     */
    void enableForeignConstraintCheck(boolean enable);

    /**
     * 备份表
     *
     * @param tableName 需要备份的表
     * @param backupTableName 创建的备份表名称
     */
    void backupTable(String tableName, String backupTableName);

    /**
     * 删除数据库表信息缓存
     * @param tableNames 需要删除的数据库表列表,为空表示全部删除
     *
     */
    void deleteDatabaseEntityCache(String... tableNames);

}
