package cn.schoolwow.quickdao.util;

import cn.schoolwow.quickdao.domain.external.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Collection;

public class ParametersUtil {
    private static Logger logger = LoggerFactory.getLogger(ParametersUtil.class);
    /**
     * SQL参数占位符
     */
    private static final String PLACEHOLDER = "** NOT SPECIFIED **";
    /**
     * 格式化旧版本的java.sql.Date类型
     */
    private final static SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
    /**
     * 格式化旧版本的java.sql.Time类型
     */
    private final static SimpleDateFormat simpleTimeFormat = new SimpleDateFormat("HH:mm:ss");
    /**
     * 格式化旧版本的Timestampt类型
     */
    private final static SimpleDateFormat simpleDateTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss:SSS");
    /**
     * 格式化日期参数
     */
    private final static DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    /**
     * 格式化日期参数
     */
    private final static DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * 设置字段值为当前日期
     *
     * @param property 字段属性
     * @param instance 实例
     */
    public static void setCurrentDateTime(Property property, Object instance) {
        Field field = getFieldFromInstance(instance, property.name);
        try {
            Object value = field.get(instance);
            //当用户手动设置该字段时,则程序不再注入时间
            if (null != value) {
                return;
            }
            switch (property.className) {
                case "java.util.Date": {
                    field.set(instance, new java.util.Date(System.currentTimeMillis()));
                }
                break;
                case "java.sql.Date": {
                    field.set(instance, new java.sql.Date(System.currentTimeMillis()));
                }
                break;
                case "java.sql.Timestamp": {
                    field.set(instance, new Timestamp(System.currentTimeMillis()));
                }
                break;
                case "java.util.Calendar": {
                    field.set(instance, Calendar.getInstance());
                }
                break;
                case "java.time.LocalDate": {
                    field.set(instance, LocalDate.now());
                }
                break;
                case "java.time.LocalDateTime": {
                    field.set(instance, LocalDateTime.now());
                }
                break;
                default: {
                    throw new IllegalArgumentException("不支持该日期类型,目前支持的类型为Date,Calendar,LocalDate,LocalDateTime!当前类型:" + property.className);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("设置当前时间异常", e);
        }
    }

    /**
     * 替换SQL语句
     *
     * @param sql        语句
     * @param parameters 参数
     */
    public static String getFormatSQL(String sql, Collection parameters) {
        if(null==parameters||parameters.isEmpty()){
            return sql;
        }
        //判断参数个数是否匹配
        int parameterCount = 0;
        for(int i=0;i<sql.length();i++){
            if(sql.charAt(i)=='?'){
                parameterCount++;
            }
        }
        if(parameterCount!=parameters.size()){
            logger.warn("参数个数不匹配!预期参数个数:{},实际参数个数:{},\r\n语句:{}\r\n参数列表:{}", parameterCount, parameters.size(), sql, parameters);
            throw new IllegalArgumentException("参数个数不匹配!预期个数:"+parameterCount+",实际个数:"+parameters.size());
        }
        StringBuilder sqlBuilder = new StringBuilder(sql.replace("?", PLACEHOLDER));
        for (Object parameter : parameters) {
            String parameterPlaceHolder = getParameterPlaceHolder(parameter);
            int indexOf = sqlBuilder.indexOf(PLACEHOLDER);
            sqlBuilder.replace(indexOf, indexOf + PLACEHOLDER.length(), parameterPlaceHolder);
        }
        String formatSQL = sqlBuilder.toString();
        return formatSQL;
    }

    /**
     * 从实例从获取字段值
     *
     * @param instance 实例
     * @param name 字段名
     */
    public static Object getFieldValueFromInstance(Object instance, String name) {
        Field field = getFieldFromInstance(instance, name);
        try {
            Object value = field.get(instance);
            return value;
        } catch (IllegalAccessException e) {
            throw new RuntimeException("获取实例字段值失败", e);
        }
    }

    /**
     * 从实例从获取字段信息
     *
     * @param instance 实例
     * @param name 字段名
     */
    public static Field getFieldFromInstance(Object instance, String name) {
        Class tempClass = instance.getClass();
        Field field = null;
        while (null == field && null != tempClass) {
            Field[] fields = tempClass.getDeclaredFields();
            for (Field field1 : fields) {
                if (field1.getName().equals(name)) {
                    field = field1;
                    break;
                }
            }
            tempClass = tempClass.getSuperclass();
        }
        if (null == field) {
            throw new IllegalArgumentException("字段不存在!字段名:" + name + ",类名:" + instance.getClass().getName());
        }
        field.setAccessible(true);
        return field;
    }

    /**
     * 获取参数占位符
     *
     * @param parameter 参数值
     */
    private static String getParameterPlaceHolder(Object parameter) {
        if (null == parameter) {
            return "null";
        }
        String parameterSQL = parameter.toString();
        switch (parameter.getClass().getName()) {
            case "boolean": {
                boolean value = (boolean) parameter;
                parameterSQL = value ? "1" : "0";
            }
            break;
            case "java.lang.String":
            case "com.alibaba.fastjson.JSONObject":
            case "com.alibaba.fastjson.JSONArray": {
                parameterSQL = "'" + parameter.toString() + "'";
            }
            break;
            case "java.util.Date": {
                java.util.Date date = (java.util.Date) parameter;
                parameterSQL = "'" + simpleDateFormat.format(date) + "'";
            }
            break;
            case "java.sql.Date": {
                Date date = (Date) parameter;
                parameterSQL = "'" + simpleDateFormat.format(date) + "'";
            }
            break;
            case "java.sql.Time": {
                Time time = (Time) parameter;
                parameterSQL = "'" + simpleTimeFormat.format(time) + "'";
            }
            break;
            case "java.sql.Timestamp": {
                Timestamp timestamp = (Timestamp) parameter;
                parameterSQL = "'" + simpleDateTimeFormat.format(timestamp) + "'";
            }
            break;
            case "java.time.LocalDate": {
                LocalDate localDate = (LocalDate) parameter;
                parameterSQL = "'" + localDate.format(dateFormatter) + "'";
            }
            break;
            case "java.time.LocalDateTime": {
                LocalDateTime localDateTime = (LocalDateTime) parameter;
                parameterSQL = "'" + localDateTime.format(dateTimeFormatter) + "'";
            }
            break;
        }
        return parameterSQL;
    }
}
