package cn.schoolwow.quickdao.dao.transaction;

import cn.schoolwow.quickdao.dao.dml.DatabaseManipulationImpl;
import cn.schoolwow.quickdao.dao.query.QueryOperation;
import cn.schoolwow.quickdao.dao.query.QueryOperationImpl;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.exception.SQLRuntimeException;
import cn.schoolwow.quickdao.flow.handler.QuickDAOTryCatchFinallyHandler;
import cn.schoolwow.quickflow.QuickFlow;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;

/**
 * 事务操作代理模式对象
 */
public class TransactionInvocationHandler implements InvocationHandler {
    private Connection transactionConnection;
    private QuickFlow quickFlow;
    private QuickDAOConfig quickDAOConfig;
    private QueryOperation queryOperation;
    private TransactionOperation transactionOperation;

    public TransactionInvocationHandler(QuickDAOConfig quickDAOConfig) {
        try {
            this.quickFlow = setTransactionConnection(quickDAOConfig);
            this.quickDAOConfig = quickDAOConfig;
            this.queryOperation = new QueryOperationImpl(quickFlow, quickDAOConfig);
            this.transactionOperation = new TransactionOperationImpl(transactionConnection);
        }catch (SQLException e){
            throw new SQLRuntimeException(e);
        }
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        String interfaceName = method.getDeclaringClass().getSimpleName();
        switch (interfaceName) {
            case "Closeable":
            case "TransactionOperation": {
                return method.invoke(transactionOperation, args);
            }
            case "Object":
            case "DatabaseDAO":
            case "DatabaseManipulation": {
                return method.invoke(new DatabaseManipulationImpl(quickFlow, quickDAOConfig), args);
            }
            case "QueryOperation":{
                return method.invoke(queryOperation, args);
            }
            default: {
                throw new IllegalArgumentException("不支持的方法调用!接口名:"+interfaceName+",方法:" + method);
            }
        }
    }

    private QuickFlow setTransactionConnection(QuickDAOConfig quickDAOConfig) throws SQLException {
        transactionConnection = quickDAOConfig.databaseContext.dataSource.getConnection();
        transactionConnection.setAutoCommit(false);

        QuickFlow quickFlow = QuickFlow.newInstance()
                .printTrace(quickDAOConfig.logRecordOption.recordFlowLog)
                .tryCatchFinallyHandler(new QuickDAOTryCatchFinallyHandler(transactionConnection, quickDAOConfig));
        return quickFlow;
    }
}
