package cn.schoolwow.quickdao.dao.ddl;

import cn.schoolwow.quickdao.dao.sql.DatabaseDAOImpl;
import cn.schoolwow.quickdao.domain.external.*;
import cn.schoolwow.quickdao.flow.ddl.common.DeleteDatabaseEntityCacheFlow;
import cn.schoolwow.quickdao.flow.ddl.createTable.CreateTableFlow;
import cn.schoolwow.quickdao.flow.ddl.has.HasConstraintFlow;
import cn.schoolwow.quickdao.flow.ddl.has.HasIndexFlow;
import cn.schoolwow.quickdao.flow.ddl.has.HasTableColumnFlow;
import cn.schoolwow.quickdao.flow.ddl.has.HasTableFlow;
import cn.schoolwow.quickdao.flow.ddl.index.*;
import cn.schoolwow.quickdao.flow.ddl.property.*;
import cn.schoolwow.quickdao.flow.ddl.table.*;
import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

public class DatabaseDefinitionImpl extends DatabaseDAOImpl implements DatabaseDefinition {
    protected Logger logger = LoggerFactory.getLogger(DatabaseDefinitionImpl.class);

    public DatabaseDefinitionImpl(QuickFlow quickFlow, QuickDAOConfig quickDAOConfig) {
        super(quickFlow, quickDAOConfig);
    }

    @Override
    public boolean hasTable(Class clazz) {
        Entity entity = quickDAOConfig.getEntityByClassName(clazz.getName());
        return hasTable(entity.tableName);
    }

    @Override
    public boolean hasTable(String tableName) {
        FlowContext flowContext = quickFlow.startFlow(new HasTableFlow())
                .putTemporaryData("tableName", tableName)
                .execute();
        boolean hasResult = (boolean) flowContext.checkData("hasResult");
        return hasResult;
    }

    @Override
    public boolean hasColumn(Class clazz, String columnName) {
        FlowContext flowContext = quickFlow.startFlow(new HasTableColumnFlow())
                .putTemporaryData("clazz", clazz)
                .putTemporaryData("columnName", columnName)
                .execute();
        boolean hasResult = (boolean) flowContext.checkData("hasResult");
        return hasResult;
    }

    @Override
    public boolean hasColumn(String tableName, String columnName) {
        FlowContext flowContext = quickFlow.startFlow(new HasTableColumnFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("columnName", columnName)
                .execute();
        boolean hasResult = (boolean) flowContext.checkData("hasResult");
        return hasResult;
    }

    @Override
    public List<String> getTableNameList(){
        FlowContext flowContext = quickFlow.executeFlow(new GetTableNameListFlow());
        List<String> tableNameList = (List<String>) flowContext.getData("tableNameList");
        return tableNameList;
    }

    @Override
    public List<Entity> getDatabaseEntityList() {
        FlowContext flowContext = quickFlow.startFlow("获取数据库所有表信息")
                .next(new GetEntityListFlow())
                .next(new GetEntityPropertyListFlow())
                .next(new GetEntityIndexListFlow())
                .execute();
        List<Entity> entityList = (List<Entity>) flowContext.getData("entityList");
        return entityList;
    }

    @Override
    public Entity getDatabaseEntity(String tableName) {
        FlowContext flowContext = quickFlow.startFlow("获取数据库指定表信息")
                .putTemporaryData("tableName", tableName)
                .next(new GetEntityFlow())
                .next(new GetEntityPropertyFlow())
                .next(new GetEntityIndexFlow())
                .execute();
        Entity entity = (Entity) flowContext.getData("entity");
        return entity;
    }

    @Override
    public List<Property> getPropertyList(Class clazz) {
        Entity entity = quickDAOConfig.getEntityByClassName(clazz.getName());
        return getPropertyList(entity.tableName);
    }

    @Override
    public List<Property> getPropertyList(String tableName) {
        Entity entity = new Entity();
        entity.tableName = tableName;
        quickFlow.startFlow(new GetEntityPropertyFlow())
                .putTemporaryData("entity", entity)
                .execute();
        return entity.properties;
    }

    @Override
    public Property getProperty(Class clazz, String columnName) {
        FlowContext flowContext = quickFlow.startFlow(new GetPropertyFlow())
                .putTemporaryData("clazz", clazz)
                .putTemporaryData("columnName", columnName)
                .execute();
        Property property = (Property) flowContext.getData("property");
        if(null==property){
            throw new IllegalArgumentException("指定表的指定字段不存在!实体类:"+clazz.getName()+",字段名:"+columnName);
        }
        return property;
    }

    @Override
    public Property getProperty(String tableName, String columnName) {
        FlowContext flowContext = quickFlow.startFlow(new GetPropertyFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("columnName", columnName)
                .execute();
        Property property = (Property) flowContext.getData("property");
        if(null==property){
            throw new IllegalArgumentException("指定表的指定字段不存在!表名:"+tableName+",字段名:"+columnName);
        }
        return property;
    }

    @Override
    public void create(Class clazz) {
        Entity entity = quickDAOConfig.getEntityByClassName(clazz.getName());
        create(entity);
    }

    @Override
    public void create(Entity entity) {
        quickFlow.startFlow(new CreateTableFlow())
                .putTemporaryData("entity", entity)
                .execute();
    }

    @Override
    public void renameTable(String oldTableName, String newTableName) {
        quickFlow.startFlow(new RenameTableFlow())
                .putTemporaryData("oldTableName", oldTableName)
                .putTemporaryData("newTableName", newTableName)
                .execute();
    }

    @Override
    public void dropTable(Class clazz) {
        Entity entity = quickDAOConfig.getEntityByClassName(clazz.getName());
        dropTable(entity.tableName);
    }

    @Override
    public void dropTable(String tableName) {
        quickFlow.startFlow(new DropTableFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("tableName", tableName)
                .execute();
    }

    @Override
    public void rebuild(Class clazz) {
        if (hasTable(clazz)) {
            dropTable(clazz);
        }
        create(clazz);
    }

    @Override
    public void rebuild(String tableName) {
        if(hasTable(tableName)){
            dropTable(tableName);
        }
        Entity entity = quickDAOConfig.getDatabaseEntityByTableName(tableName);
        create(entity);
    }

    @Override
    public Property createColumn(String tableName, PropertyOption propertyOption) {
        FlowContext flowContext = quickFlow.startFlow("新增表字段")
                .ignoreBroken(false)
                .next(new HasTableFlow())
                .next(new CreateTableColumnFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .next(new GetPropertyFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("propertyOption", propertyOption)
                .putTemporaryData("columnName", propertyOption.column)
                .execute();
        Property property = (Property) flowContext.checkData("property");
        return property;
    }

    @Override
    public void renameTableColumn(String tableName, String oldTableColumn, String newTableColumn) {
        quickFlow.startFlow(new RenameTableColumnFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("oldTableColumn", oldTableColumn)
                .putTemporaryData("newTableColumn", newTableColumn)
                .execute();
    }

    @Override
    public void alterColumn(String tableName, PropertyOption propertyOption) {
        quickFlow.startFlow(new AlterTableColumnFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("propertyOption", propertyOption)
                .execute();
    }

    @Override
    public Property dropColumn(String tableName, String columnName) {
        FlowContext flowContext = quickFlow.startFlow(new DropTableColumnFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("columnName", columnName)
                .execute();
        Property property = (Property) flowContext.checkData("property");
        return property;
    }

    @Override
    public boolean hasIndex(String tableName, String indexName) {
        FlowContext flowContext = quickFlow.startFlow("判断索引是否存在")
                .ignoreBroken(false)
                .next(new HasTableFlow())
                .next(new HasIndexFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("indexName", indexName)
                .execute();
        boolean hasResult = (boolean) flowContext.checkData("hasResult");
        return hasResult;
    }

    @Override
    public boolean hasConstraint(String tableName, String constraintName) {
        FlowContext flowContext = quickFlow.startFlow(new HasConstraintFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("constraintName", constraintName)
                .execute();
        boolean hasResult = (boolean) flowContext.checkData("hasResult");
        return hasResult;
    }

    @Override
    public List<IndexField> getIndexFieldList(String tableName) {
        Entity entity = new Entity();
        entity.tableName = tableName;
        quickFlow.startFlow(new GetEntityIndexFlow())
                .putTemporaryData("entity", entity)
                .execute();
        for(IndexField indexField:entity.indexFieldList){
            indexField.tableName = tableName;
        }
        return entity.indexFieldList;
    }

    @Override
    public IndexField getIndexField(String tableName, String indexName) {
        FlowContext flowContext = quickFlow.startFlow(new GetSingleEntityIndexFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("indexName", indexName)
                .execute();
        return (IndexField) flowContext.getData("indexField");
    }

    @Override
    public void createIndex(IndexField indexField) {
        quickFlow.startFlow(new CreateIndexFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("indexField", indexField)
                .putTemporaryData("tableName", indexField.tableName)
                .execute();
    }

    @Override
    public void dropIndex(String tableName, String indexName) {
        quickFlow.startFlow(new DropIndexFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("indexName", indexName)
                .execute();
    }

    @Override
    public void createForeignKey(String tableName, PropertyOption propertyOption) {
        quickFlow.startFlow(new CreateForeignKeyFlow())
                .next(new DeleteDatabaseEntityCacheFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("propertyOption", propertyOption)
                .execute();
    }

    @Override
    public void enableForeignConstraintCheck(boolean enable) {
        quickFlow.startFlow(new EnableForeignConstraintCheckFlow())
                .putTemporaryData("enable", enable)
                .execute();
    }

    @Override
    public void backupTable(String tableName, String backupTableName) {
        quickFlow.startFlow(new BackupTableFlow())
                .putTemporaryData("tableName", tableName)
                .putTemporaryData("backupTableName", backupTableName)
                .execute();
    }
}