package cn.schoolwow.quickdao.dao;

import cn.schoolwow.quickdao.dao.dql.condition.AbstractCondition;
import cn.schoolwow.quickdao.dao.dql.condition.Condition;
import cn.schoolwow.quickdao.dao.transaction.Transaction;
import cn.schoolwow.quickdao.dao.transaction.TransactionInvocationHandler;
import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.QuickDAOConfig;
import cn.schoolwow.quickdao.domain.internal.Query;
import cn.schoolwow.quickdao.provider.DatabaseProvider;
import cn.schoolwow.quickdao.statement.dql.response.GetResponseArrayDatabaseStatement;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.lang.reflect.Proxy;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

public class DAOOperationImpl implements DAOOperation {
    private Logger logger = LoggerFactory.getLogger(DAOOperationImpl.class);

    private final QuickDAOConfig quickDAOConfig;

    public DAOOperationImpl(QuickDAOConfig quickDAOConfig) {
        this.quickDAOConfig = quickDAOConfig;
    }

    @Override
    public <T> Condition<T> query(Class<T> clazz) {
        Entity entity = quickDAOConfig.getEntityByClassName(clazz.getName());
        if (null == entity) {
            throw new IllegalArgumentException("不存在的实体类:" + clazz.getName() + "!");
        }
        return query(entity);
    }

    @Override
    public Condition query(String tableName) {
        Entity entity = quickDAOConfig.getDatabaseEntityByTableName(tableName);
        if (null == entity) {
            for (String virtualTableName : quickDAOConfig.databaseOption.virtualTableNameList) {
                if (virtualTableName.equalsIgnoreCase(tableName)) {
                    entity = new Entity();
                    entity.tableName = virtualTableName;
                    entity.properties = new ArrayList<>();
                    break;
                }
            }
        }
        if (null == entity) {
            throw new IllegalArgumentException("不存在的表名:" + tableName + "!");
        }
        return query(entity);
    }

    @Override
    public Condition query(Condition condition) {
        Query fromQuery = ((AbstractCondition) condition).query;
        condition.execute();

        Entity entity = new Entity();
        entity.clazz = JSONObject.class;
        entity.properties = new ArrayList<>();
        AbstractCondition condition1 = (AbstractCondition) query(entity);
        condition1.query.fromQuery = fromQuery;
        entity.tableName = "( " + new GetResponseArrayDatabaseStatement(fromQuery).getStatement() + ")";
        return condition1;
    }

    @Override
    public Transaction getThreadTransaction() {
        Transaction transactionProxy = quickDAOConfig.transactionThreadLocal.get();
        return transactionProxy;
    }

    @Override
    public Transaction startTransaction() {
        TransactionInvocationHandler transactionInvocationHandler = new TransactionInvocationHandler(quickDAOConfig);
        Transaction transactionProxy = (Transaction) Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class<?>[]{Transaction.class}, transactionInvocationHandler);
        quickDAOConfig.transactionThreadLocal.set(transactionProxy);
        return transactionProxy;
    }

    @Override
    public void startTransaction(Consumer<Transaction> transactionConsumer) {
        Transaction transaction = startTransaction();
        try {
            transactionConsumer.accept(transaction);
            transaction.commit();
        } catch (Exception e) {
            transaction.rollback();
            throw new RuntimeException(e);
        } finally {
            transaction.close();
        }
    }

    @Override
    public DataSource getDataSource() {
        return quickDAOConfig.dataSource;
    }

    @Override
    public Map<String, Entity> getEntityMap() {
        return quickDAOConfig.entityMap;
    }

    @Override
    public Entity getEntity(Class clazz) {
        return quickDAOConfig.entityMap.values().stream().filter(entity -> entity.clazz.getName().equalsIgnoreCase(clazz.getName())).findFirst().orElse(null);
    }

    @Override
    public Entity getEntity(String tableName) {
        return quickDAOConfig.entityMap.values().stream().filter(entity -> entity.tableName.equalsIgnoreCase(tableName)).findFirst().orElse(null);
    }

    @Override
    public DatabaseProvider getDatabaseProvider() {
        return quickDAOConfig.databaseProvider;
    }

    @Override
    public QuickDAOConfig getQuickDAOConfig() {
        return quickDAOConfig;
    }

    @Override
    public void startRecord() {
        quickDAOConfig.record = true;
    }

    @Override
    public String stopRecord() {
        quickDAOConfig.record = false;
        String sqlRecord = quickDAOConfig.sqlRecordBuilder.toString();
        quickDAOConfig.sqlRecordBuilder.setLength(0);
        return sqlRecord;
    }

    @Override
    public <T> T log(Class<T> logEntity) {
        return logMap("#SINGLE_INSTANCE#", logEntity);
    }

    @Override
    public <T> T logMapIfExist(String uniqueKey, Class<T> logEntity) {
        try {
            if(null==quickDAOConfig.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(classNameLogInstance.containsKey(uniqueKey)){
                T logInstance = (T) classNameLogInstance.get(uniqueKey);
                return logInstance;
            }
            return null;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public <T> List<T> logMap(Class<T> logEntity) {
        try {
            if(null==quickDAOConfig.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(null==classNameLogInstance){
                return null;
            }
            List<T> instanceList = new ArrayList<>();
            for(Object instance:classNameLogInstance.values()){
                instanceList.add((T) instance);
            }
            return instanceList;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public <T> T logMap(String uniqueKey, Class<T> logEntity) {
        try {
            if(null==quickDAOConfig.classNameLogInstanceMapThreadLocal.get()){
                Map<String,Map<String,Object>> classNameLogInstanceMap = new HashMap<>();
                quickDAOConfig.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
            }
            Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.classNameLogInstanceMapThreadLocal.get();
            if(!classNameLogInstanceMap.containsKey(logEntity.getName())){
                classNameLogInstanceMap.put(logEntity.getName(), new ConcurrentHashMap<>());
            }
            Map<String,Object> classNameLogInstance = classNameLogInstanceMap.get(logEntity.getName());
            if(!classNameLogInstance.containsKey(uniqueKey)){
                classNameLogInstance.put(uniqueKey, logEntity.newInstance());
            }
            T logInstance = (T) classNameLogInstance.get(uniqueKey);
            return logInstance;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    @Override
    public void insertLog() {
        if(null==quickDAOConfig.classNameLogInstanceMapThreadLocal.get()){
            Map<String,Map<String,Object>> classNameLogInstanceMap = new ConcurrentHashMap<>();
            quickDAOConfig.classNameLogInstanceMapThreadLocal.set(classNameLogInstanceMap);
        }
        Map<String,Map<String,Object>> classNameLogInstanceMap = quickDAOConfig.classNameLogInstanceMapThreadLocal.get();
        if(null==classNameLogInstanceMap){
            return;
        }
        Collection<Map<String,Object>> logInstanceMapCollection = classNameLogInstanceMap.values();
        for(Map<String,Object> logInstanceMap:logInstanceMapCollection){
            Collection<Object> logInstances = logInstanceMap.values();
            quickDAOConfig.dao.insert(logInstances);
        }
        classNameLogInstanceMap.clear();
    }

    /**
     * 数据库查询
     */
    private Condition query(Entity entity) {
        Query query = new Query();
        query.entity = entity;
        query.quickDAOConfig = quickDAOConfig;
        return quickDAOConfig.databaseProvider.getConditionInstance(query);
    }
}