package cn.schoolwow.ams.util;

import cn.schoolwow.quickdao.annotation.Comment;
import cn.schoolwow.quickdao.dao.dql.condition.Condition;
import cn.schoolwow.quickdao.domain.external.Entity;
import cn.schoolwow.quickdao.domain.external.PageVo;
import cn.schoolwow.quickdao.domain.external.Property;
import cn.schoolwow.quickdao.util.StringUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

public class AMSUtil {
    private static Logger logger = LoggerFactory.getLogger(AMSUtil.class);

    /**获取指定目录下所有文件*/
    public static List<String> getJavascriptResourcePathList(String path) throws IOException {
        ClassPathResource resource = new ClassPathResource(path);
        if(!resource.exists()){
            return new ArrayList<>();
        }
        URL url = resource.getURI().toURL();
        String prefix = path.substring(path.lastIndexOf("/")+1);
        List<String> resourcePathList = new ArrayList<>();
        switch (url.getProtocol()) {
            case "file": {
                File directory = new File(url.getFile());
                Files.walkFileTree(directory.toPath(), new SimpleFileVisitor<Path>() {
                    @Override
                    public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) throws IOException {
                        File file = path.toFile();
                        if (file.getName().endsWith(".js")) {
                            String relativePath = file.getAbsolutePath().replace(directory.getAbsolutePath(), "").replace("\\", "/");
                            resourcePathList.add(prefix+relativePath);
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
            break;
            case "jar": {
                JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
                if (null != jarURLConnection) {
                    JarFile jarFile = jarURLConnection.getJarFile();
                    if (null != jarFile) {
                        Enumeration<JarEntry> jarEntries = jarFile.entries();
                        while (jarEntries.hasMoreElements()) {
                            JarEntry jarEntry = jarEntries.nextElement();
                            String jarEntryName = jarEntry.getName();
                            if(jarEntryName.startsWith("static")&&jarEntryName.contains(path)&&jarEntryName.endsWith(".js")){
                                resourcePathList.add(jarEntryName.substring(jarEntryName.indexOf("/")+1));
                            }
                        }
                    }
                }
            }
            break;
        }
        return resourcePathList;
    }

    /**分页和排序*/
    public static Condition pageAndSort(Condition condition, JSONObject amsListCondition) {
        //分页
        {
            int page = amsListCondition.containsKey("page")?amsListCondition.getInteger("page"):1;
            int pageSize = amsListCondition.containsKey("pageSize")?amsListCondition.getInteger("pageSize"):10;
            condition.page(page, pageSize);
        }
        //排序
        {
            String sortField = amsListCondition.getString("sortField");
            String sortOrder = amsListCondition.getString("sortOrder");
            if (StringUtils.isNoneBlank(sortField) && StringUtils.isNoneBlank(sortOrder)) {
                condition.order(sortField, sortOrder);
            }
        }
        amsListCondition.remove("page");
        amsListCondition.remove("pageSize");
        amsListCondition.remove("sortField");
        amsListCondition.remove("sortOrder");
        return condition;
    }

    /**添加数据库字段复杂查询*/
    public static void addCompositeQuery(Entity entity, Condition condition, JSONObject amsListCondition){
        Set<String> keySet = amsListCondition.keySet();
        for(String key:keySet){
            Property property = entity.getPropertyByFieldName(key);
            if(null==property){
                continue;
            }
            String value = amsListCondition.getString(key);
            condition.addCompositeQuery(key, value);
        }
    }

    /**封装list结果*/
    public static JSONObject getAMSList(List list) {
        JSONObject amsList = new JSONObject();
        amsList.put("list", list);
        return amsList;
    }

    /**封装list结果*/
    public static JSONObject getAMSPagingList(PageVo pageVo) {
        JSONObject amsList = new JSONObject();
        amsList.put("list", pageVo.getArray());
        amsList.put("total", pageVo.getTotalSize());
        return amsList;
    }

    /**添加数据库表区块*/
    public static void addDatabaseEntityBlock(String daoName, Entity entity, JSONArray blocks){
        JSONObject block = new JSONObject();
        block.put("daoName", daoName);
        block.put("tableName", entity.tableName);
        //字段
        {
            List<Property> propertyList = entity.properties;
            JSONObject resourceFields = new JSONObject(true);
            JSONObject operationFields = new JSONObject(true);
            for (Property property : propertyList) {
                switch (property.column) {
                    case "created_at": {
                        property.comment = "创建时间";
                    }
                    break;
                    case "updated_at": {
                        property.comment = "更新时间";
                    }
                    break;
                }
                if (!property.column.equalsIgnoreCase("id") && property.comment.isEmpty()) {
                    continue;
                }
                JSONObject field = new JSONObject();
                if (property.column.equalsIgnoreCase("id")) {
                    property.comment = "id";
                }
                //判断是否下拉选择
                if (property.comment.contains("(")) {
                    String comment = property.comment.substring(0, property.comment.indexOf("("));
                    field.put("label", comment);
                    field.put("info", "请选择" + comment);
                    field.put("type", "select");

                    JSONObject props = new JSONObject();
                    field.put("props", props);
                    props.put("placeholder", "请选择" + comment);
                    props.put("clearable", true);
                    props.put("multiple", false);

                    JSONObject options = new JSONObject();
                    props.put("options", options);

                    String selectText = property.comment.substring(property.comment.indexOf("(") + 1, property.comment.indexOf(")"));
                    String[] tokens = selectText.split(",");
                    for (String token : tokens) {
                        int indexOf = token.indexOf(":");
                        if (indexOf < 0) {
                            throw new IllegalArgumentException("实体类注释异常!表名:" + entity.tableName + ",字段名:" + property.name + ",注释:" + property.comment);
                        }
                        String key = token.substring(0, indexOf);
                        String value = token.substring(indexOf + 1);
                        options.put(key, value + "(" + key + ")");
                    }
                } else {
                    field.put("label", property.comment);
                    field.put("info", "请输入" + property.comment);
                    if (property.columnType.contains("date") || property.columnType.contains("timestamp")) {
                        field.put("type", "datetime");
                    } else if(property.comment.startsWith("是否")){
                        field.put("type", "switch");
                    }else if(property.column.contains("url")){
                        field.put("type", "link");
                    }else{
                        field.put("type", "text");
                    }

                    JSONObject props = new JSONObject();
                    field.put("props", props);
                    props.put("placeholder", "请输入" + property.comment);
                }
                String label = StringUtil.underline2Camel(property.column);
                resourceFields.put(label, field);

                JSONObject operation = new JSONObject();
                operation.put("slot", "searchs");
                operation.put("type", "field");
                operation.put("label", field.getString("label"));
                operation.put("field", field);
                operationFields.put(label, operation);
            }
            block.put("resourceFields", resourceFields);
            block.put("operationFields", operationFields);
        }
        blocks.add(block);
    }

    /**根据pojo类添加区块*/
    public static void addClassBlock(Class clazz, JSONArray blocks) {
        addClassBlock(clazz.getSimpleName(), clazz, clazz, blocks);
    }

    /**根据pojo类添加区块*/
    public static void addClassBlock(String tableName, Class requestClass, Class responseClass, JSONArray blocks) {
        JSONObject block = new JSONObject();
        block.put("tableName", tableName);
        block.put("daoName", "clazz");
        //请求字段
        {
            JSONObject fieldMap = new JSONObject(true);
            Field[] entityFields = requestClass.getDeclaredFields();
            Field.setAccessible(entityFields, true);
            for (Field entityField : entityFields) {
                JSONObject field = getAMSField(requestClass, entityField);
                JSONObject operation = new JSONObject();
                operation.put("slot", "searchs");
                operation.put("type", "field");
                operation.put("label", field.getString("label"));
                operation.put("field", field);
                fieldMap.put(entityField.getName(), operation);
            }
            block.put("operationFields", fieldMap);
        }
        //响应字段
        {
            JSONObject fieldMap = new JSONObject(true);
            Field[] entityFields = responseClass.getDeclaredFields();
            Field.setAccessible(entityFields, true);
            for (Field entityField : entityFields) {
                JSONObject field = getAMSField(responseClass, entityField);
                fieldMap.put(entityField.getName(), field);
            }
            block.put("resourceFields", fieldMap);
        }
        blocks.add(block);
    }

    private static JSONObject getAMSField(Class clazz, Field entityField){
        JSONObject field = new JSONObject();
        String comment = entityField.getAnnotation(Comment.class).value();
        //判断是否下拉选择
        if (comment.contains("(")) {
            String fieldComment = comment.substring(0, comment.indexOf("("));
            field.put("label", fieldComment);
            field.put("info", "请选择" + fieldComment);
            field.put("type", "select");

            JSONObject props = new JSONObject();
            field.put("props", props);
            props.put("placeholder", "请选择" + fieldComment);
            props.put("clearable", true);
            props.put("multiple", false);

            JSONObject options = new JSONObject();
            props.put("options", options);

            String selectText = comment.substring(comment.indexOf("(") + 1, comment.indexOf(")"));
            String[] tokens = selectText.split(",");
            for (String token : tokens) {
                int indexOf = token.indexOf(":");
                if (indexOf < 0) {
                    throw new IllegalArgumentException("实体类注释异常!类名:" + clazz.getName() + ",字段名:" + entityField.getName() + ",注释:" + comment);
                }
                String key = token.substring(0, indexOf);
                String value = token.substring(indexOf + 1);
                options.put(key, value + "(" + key + ")");
            }
        } else {
            field.put("label", comment);
            field.put("info", "请输入" + comment);

            switch (entityField.getType().getName()) {
                case "java.util.Date":
                case "java.sql.Timestamp":
                case "java.sql.LocalDate":
                case "java.sql.LocalDateTime": {
                    field.put("type", "datetime");
                }
                break;
                default: {
                    field.put("type", "text");
                }
            }
            JSONObject props = new JSONObject();
            field.put("props", props);
            props.put("placeholder", "请输入" + comment);
        }
        return field;
    }

}
