/*
 * Copyright (c) 2020 Renlm
 * MyExcel is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 * 	http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */
package cn.renlm.plugins.MyExcel.handler;

import java.util.Date;
import java.util.Map;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import cn.renlm.plugins.MyExcel.config.MyColumn;
import cn.renlm.plugins.MyExcel.config.MySheet;
import cn.renlm.plugins.MyExcel.config.column.Dict.DictItem;
import cn.renlm.plugins.MyExcel.config.column.Dict.DictType;
import cn.renlm.plugins.MyExcel.entity.CheckResult;

/**
 * 行数据读取处理
 * 
 * @author RenLiMing(任黎明)
 *
 */
@FunctionalInterface
public interface DataReadHandler {

	/**
	 * 单行数据处理
	 * 
	 * @param data
	 * @param checkResult
	 */
	void handle(final Map<String, Object> data, CheckResult checkResult);

	/**
	 * 单行数据读取转换
	 * 
	 * @param sheet
	 * @param rowIndex
	 * @param data
	 * @return
	 */
	default CheckResult readConvert(MySheet sheet, long rowIndex, Map<String, Object> data) {
		CheckResult checkResult = new CheckResult().setRowIndex(rowIndex).setProcess(true);
		Map<String, MyColumn> fieldColMap = sheet.getFieldColMap();
		for (Map.Entry<String, MyColumn> entry : fieldColMap.entrySet()) {
			String field = entry.getKey();
			MyColumn col = entry.getValue();
			// 空值
			if (StrUtil.isBlankIfStr(data.get(field))) {
				data.put(field, null);
			}
			// 去除前缀
			if (data.get(field) != null && StrUtil.isNotBlank(col.getPrefix())
					&& data.get(field).toString().startsWith(col.getPrefix())) {
				String valStr = data.get(field).toString();
				data.put(field, StrUtil.removePrefix(valStr, col.getPrefix()));
			}
			// 去除后缀
			if (data.get(field) != null && StrUtil.isNotBlank(col.getSuffix())
					&& data.get(field).toString().endsWith(col.getSuffix())) {
				String valStr = data.get(field).toString();
				data.put(field, StrUtil.removeSuffix(valStr, col.getSuffix()));
			}
			// 字典转换
			if (data.get(field) != null && col.getDict() != null) {
				String valStr = data.get(field).toString();
				if (col.getDict().getType() == DictType.key) {
					if (BooleanUtil.isTrue(col.getDict().isForceCheck())) {
						if (StrUtil.isNotBlank(valStr) && !col.getDict().getKeyMap().containsKey(valStr)) {
							String message = StrUtil.format("{} / 第{}行，{}，必须为下拉限定值", sheet.getName(), rowIndex,
									col.getTitle().getText());
							checkResult.getErrors().add(message);
							continue;
						}
					}
					DictItem di = col.getDict().getKeyMap().get(valStr);
					if (di != null && StrUtil.isNotBlank(col.getDict().getConvertToField())) {
						data.put(col.getDict().getConvertToField(), di.getValue());
					}
				} else if (col.getDict().getType() == DictType.value) {
					if (BooleanUtil.isTrue(col.getDict().isForceCheck())) {
						if (StrUtil.isNotBlank(valStr) && !col.getDict().getValMap().containsKey(valStr)) {
							String message = StrUtil.format("{} / 第{}行，{}，必须为下拉限定值", sheet.getName(), rowIndex,
									col.getTitle().getText());
							checkResult.getErrors().add(message);
							continue;
						}
					}
					DictItem di = col.getDict().getValMap().get(valStr);
					if (di != null) {
						valStr = di.getKey();
						data.put(field, valStr);
					}
				}
			}
			// 数字格式化
			if (data.get(field) != null && StrUtil.isNotBlank(col.getNumberFormat())) {
				String valStr = data.get(field).toString();
				data.put(field, NumberUtil.parseNumber(valStr));
			}
			// 日期转换
			if (data.get(field) != null && StrUtil.isNotBlank(col.getDateFormat())) {
				if (!(data.get(field) instanceof Date)) {
					try {
						String valStr = data.get(field).toString();
						data.put(field, DateUtil.parse(valStr, col.getDateFormat()));
					} catch (Exception e) {
						data.put(field, null);
						String message = StrUtil.format("{} / 第{}行，日期格式错误，限定{}", sheet.getName(), rowIndex,
								col.getTitle().getText(), col.getDateFormat());
						checkResult.getErrors().add(message);
					}
				}
			}
			// 非空字段
			if (col.isNotNull() && (data.get(field) == null || StrUtil.isBlankIfStr(data.get(field)))) {
				String message = StrUtil.format("{} / 第{}行，{}，不能为空", sheet.getName(), rowIndex,
						col.getTitle().getText());
				checkResult.getErrors().add(message);
			}
		}
		return checkResult;
	}
}