
package com.ds.common.swing;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.security.AccessController;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Vector;

import com.ds.common.util.FileUtility;

/**
 * <p>Description: Get internationalized resource.</p>
 * <p>Copyright: Copyright (c) 2006</p>
 * <p>Company: www.justdos.net</p>
 * @author wenzhang li
 * @version 1.0
 */
public class ResourceLocator {

    public final static char UNIX_SEPERATOR_CHAR = '/';
    public final static char WINDOWS_SEPERATOR_CHAR = '/';
    public final static String UNIX_SEPERATOR = "" + '/';
    public final static String WINDOWS_SEPERATOR = "" + '/';

    /**
     *  Maximum length of one branch of the resource search path tree. Used in
     *  getBundle.
     */
    private final static int MAX_BUNDLES_SEARCHED = 4;

    /**
     *  Gets internationalized file resource.  
     *
     *@param  resHome  directory name the resource file lies in
     *@param  resName  resource file name
     *@param  locale   Locale instance
     *@return          internationalized file resource
     */
    public static File getResourceAsFile(final String resHome, final String resName, final Locale locale) {

        if (resHome == null) {
            throw new NullPointerException("Parameter resHome can't be null.");
        }

        if (resName == null) {
            throw new NullPointerException("Parameter resName can't be null.");
        }

        if (locale == null) {
            throw new NullPointerException("Parameter locale can't be null.");
        }

        Enumeration resEnu = calculateResourceNames(getFileName(resHome, resName), locale).elements();
        File resourceFile;
        while (resEnu.hasMoreElements()) {
            resourceFile = new File((String)resEnu.nextElement());
            if (resourceFile.exists()) {
                return resourceFile;
            }
        }

        return null;
    }

    /**
     *  Gets internationalized file resource.  
     *
     *@param  resHome  directory name the resource file lies in
     *@param  resName  resource file name
     *@return          internationalized file resource
     */
    public static File getResourceAsFile(final String resHome, final String resName) {
        return getResourceAsFile(resHome, resName, Locale.getDefault());
    }

    /**
     *  Gets internationalized resource.  
     *
     *@param  resName  resource name
     *@param  locale   Locale instance
     *@return          internationalized resource
     */
    public static URL getResource(final String resName, final Locale locale) {

        if (resName == null) {
            throw new NullPointerException("Parameter resName can't be null.");
        }

        if (locale == null) {
            throw new NullPointerException("Parameter locale can't be null.");
        }

        // Identify the class loader we will be using
        final ClassLoader classLoader = getContextClassLoader();

        URL url = (URL)AccessController.doPrivileged(new java.security.PrivilegedAction() {
            public Object run() {
                Enumeration resEnu = calculateResourceNames(resName, locale).elements();
                String resourceName;
                URL result;
                while (resEnu.hasMoreElements()) {
                    resourceName = (String)resEnu.nextElement();
                    if (classLoader != null) {
                        result = classLoader.getResource(resourceName);
                        if (result != null) {
                            return result;
                        }
                    } else {
                        result = ClassLoader.getSystemResource(resourceName);
                        if (result != null) {
                            return result;
                        }
                    }

                }

                return null;
            }
        });

        return url;
    }

    /**
     *  Gets internationalized resource.  
     *
     *@param  resName  resource name
     *@return          internationalized resource
     */
    public static URL getResource(final String resName) {
        return getResource(resName, Locale.getDefault());
    }

    /**
     *  Gets internationalized resource as stream.  
     *
     *@param  resName  resource name
     *@param  locale   Locale instance
     *@return          internationalized stream resource
     */
    public static InputStream getResourceAsStream(final String resName, final Locale locale) {
        URL url = getResource(resName, locale);

        try {
            return url != null ? url.openStream() : null;
        } catch (IOException e) {
            return null;
        }

    }

    /**
     *  Gets internationalized resource as stream.  
     *
     *@param  resName  resource name
     *@return          internationalized stream resource
     */
    public static InputStream getResourceAsStream(final String resName) {
        return getResourceAsStream(resName, Locale.getDefault());
    }

    /**
     *  Return the thread context class loader if available. Otherwise return
     *  null. The thread context class loader is available for JDK 1.2 or later,
     *  if certain security conditions are met.
     *
     *@return    The contextClassLoader value
     */
    protected static ClassLoader getContextClassLoader() {
        ClassLoader classLoader = null;

        try {
            // Are we running on a JDK 1.2 or later system?
            Method method = Thread.class.getMethod("getContextClassLoader", null);

            // Get the thread context class loader (if there is one)
            classLoader = (ClassLoader)method.invoke(Thread.currentThread(), null);

        } catch (Exception e) {
            ;
        }

        // Return the selected class loader
        return classLoader;
    }

    /**
     *  Finds the resource with the given class and name.
     *
     *@param  class1    The class which will load the resource
     *@param  resName   resource name
     *@param  locale    Locale instance
     *@return           The resource value
     */
    public static URL getResource(final Class class1, final String resName, final Locale locale) {
        if (class1 == null) {
            throw new NullPointerException("Parameter class1 can't be null.");
        }

        if (resName == null) {
            throw new NullPointerException("Parameter resName can't be null.");
        }

        if (locale == null) {
            throw new NullPointerException("Parameter locale can't be null.");
        }

        String resource = resName;

        Enumeration resEnu = calculateResourceNames(resName, locale).elements();
        URL url = null;
        while (resEnu.hasMoreElements()) {
            url = class1.getResource((String)resEnu.nextElement());
            if (url != null) {
                break;
            }
        }

        if (url == null) {
            String resourceExt = "";
            String resourceName = "";
            StringBuffer buffer = new StringBuffer();
            int idx;

            if (resource.charAt(0) == '/') {
                resource = resource.substring(1, resource.length());
            } else {
                String fullClassName = class1.getName();
                idx = fullClassName.lastIndexOf('.');
                if (idx != -1) {
                    buffer.append(fullClassName.substring(0, idx));
                }
            }

            if (buffer.length() > 0) {
                buffer.append('.');
            }

            idx = resource.lastIndexOf('.');
            if (idx != -1) {
                resourceName = resource.substring(0, idx).replace('/', '.').replace('\\', '.');
                buffer.append(resourceName);
                resourceExt = resource.substring(idx, resource.length());
            } else {
                resourceName = resource.replace('/', '.').replace('\\', '.');
                buffer.append(resourceName);
            }

            idx = resourceName.lastIndexOf('.');
            if (idx != -1) {
                resourceName = resourceName.substring(idx + 1, resourceName.length());
            }

            String res = buffer.toString();

            StringBuffer buffer2 = new StringBuffer();
            buffer2.append('/');
            if (resourceExt.equals(".class")) {
                buffer2.append(res.replace('.', '/'));
            } else {
                idx = res.lastIndexOf('.');
                if (idx != -1) {
                    buffer2.append(res.substring(0, idx).replace('.', '/'));
                    buffer2.append('/');
                    buffer2.append(resourceName);
                } else {
                    buffer2.append(resourceName);
                }
            }
            buffer2.append(resourceExt);
            String resourceUrl = buffer2.toString();

            resEnu = calculateResourceNames(resourceUrl, locale).elements();
            while (resEnu.hasMoreElements()) {
                url = class1.getResource((String)resEnu.nextElement());
                if (url != null) {
                    break;
                }
            }
            if (url == null) {
                resourceUrl = resourceUrl.substring(1, resourceUrl.length()).replace('/', '\\');
                resourceUrl = '/' + resourceUrl;

                resEnu = calculateResourceNames(resourceUrl, locale).elements();
                while (resEnu.hasMoreElements()) {
                    url = class1.getResource((String)resEnu.nextElement());
                    if (url != null) {
                        break;
                    }
                }
            }
        }
        return url;
    }

    /**
     *  Finds the resource with the given class and name.
     *
     *@param  class1    The class which will load the resource
     *@param  resName   resource name
     *@return           The resource value
     */
    public static URL getResource(final Class class1, final String resName) {
        return getResource(class1, resName, Locale.getDefault());
    }

    /**
     *  Returns an input stream for reading the specified resource with the
     *  given class.
     *
     *@param  class1    The class which will load the resource
     *@param  resource  resource name
     *@param  locale    Locale instance
     *@return           The resourceAsStream value
     */
    public static InputStream getResourceAsStream(final Class class1, final String resource, final Locale locale) {
        URL url = getResource(class1, resource, locale);

        try {
            return url != null ? url.openStream() : null;
        } catch (IOException e) {

            return null;
        }

    }

    /**
     *  Returns an input stream for reading the specified resource with the
     *  given class.
     *
     *@param  class1    The class which will load the resource
     *@param  resource  resource name
     *@return           The resourceAsStream value
     */
    public static InputStream getResourceAsStream(final Class class1, final String resource) {
        return getResourceAsStream(class1, resource, Locale.getDefault());
    }
//
//    public static File getResource(final ServletContext ctx, final String res, final Locale locale) {
//        if (ctx == null) {
//            throw new NullPointerException("Parameter ctx can't be null.");
//        }
//        if (res == null) {
//            throw new NullPointerException("Parameter res can't be null.");
//        }
//        if (locale == null) {
//            throw new NullPointerException("Parameter locale can't be null.");
//        }
//
//        String basePath = ctx.getRealPath("/");
//        String resPath = getFileName(basePath, res);
//
//        Enumeration resEnu = calculateResourceNames(resPath, locale).elements();
//        File file;
//        while (resEnu.hasMoreElements()) {
//            file = new File((String)resEnu.nextElement());
//            if (file.exists()) {
//                return file;
//            }
//        }
//
//        return null;
//    }
//
//    public static File getResource(final ServletContext ctx, final String res) {
//        return getResource(ctx, res, Locale.getDefault());
//    }

    /**
     *  Calculate the resources along the search path from the resource name to
     *  the resource specified by resName and locale.
     *
     *@param  resName  the resource name
     *@param  locale   the locale
     *@return          Description of the Return Value
     */
    private static Vector calculateResourceNames(final String resName, final Locale locale) {
        final Vector result = new Vector(MAX_BUNDLES_SEARCHED);
        final String language = locale.getLanguage();
        final int languageLength = language.length();
        final String country = locale.getCountry();
        final int countryLength = country.length();
        final String variant = locale.getVariant();
        final int variantLength = variant.length();
        final String extension = FileUtility.getExtension(resName);

        result.add(resName);

        if (languageLength + countryLength + variantLength == 0) {
            //The locale is "", "", "".
            return result;
        }
        final StringBuffer temp = new StringBuffer(FileUtility.removeExtension(resName));
        temp.append('_');
        temp.append(language);
        if (languageLength > 0) {
            if (extension.length() > 0) {
                result.insertElementAt(temp.toString() + "." + extension, 0);
            } else {
                result.insertElementAt(temp.toString(), 0);
            }
        }

        if (countryLength + variantLength == 0) {
            return result;
        }
        temp.append('_');
        temp.append(country);
        if (countryLength > 0) {
            if (extension.length() > 0) {
                result.insertElementAt(temp.toString() + "." + extension, 0);
            } else {
                result.insertElementAt(temp.toString(), 0);
            }
        }

        if (variantLength == 0) {
            return result;
        }
        temp.append('_');
        temp.append(variant);
        if (extension.length() > 0) {
            result.insertElementAt(temp.toString() + "." + extension, 0);
        } else {
            result.insertElementAt(temp.toString(), 0);
        }

        return result;
    }

    private static String getFileName(final String basePath, final String path) {
        String bPath = basePath;
        String p = path;
        if (File.separatorChar == WINDOWS_SEPERATOR_CHAR) {
            bPath = bPath.replace(UNIX_SEPERATOR_CHAR, File.separatorChar);
            p = p.replace(UNIX_SEPERATOR_CHAR, File.separatorChar);
        } else if (File.separatorChar == UNIX_SEPERATOR_CHAR) {
            bPath = bPath.replace(WINDOWS_SEPERATOR_CHAR, File.separatorChar);
            p = p.replace(WINDOWS_SEPERATOR_CHAR, File.separatorChar);
        }

        if (bPath.endsWith(File.separator)) {
            bPath = bPath.substring(0, bPath.length() - 1);
        }
        if (!p.startsWith(File.separator)) {
            p = File.separator + p;
        }

        return basePath + path;
    }

}
